#!/usr/bin/python3

import unittest
import sys
import subprocess
from pathlib import Path
from tempfile import NamedTemporaryFile


class TmpFile:
    def __init__(self, content):
        self.content = content

    def __enter__(self):
        self.file = NamedTemporaryFile(mode="w+")
        self.file.write(self.content)
        self.file.flush()
        return self.file.name

    def __exit__(self, type, value, traceback):
        self.file.close()


class WrapTestCase(unittest.TestCase):
    @staticmethod
    def exec_wrap(args: [str] = []) -> subprocess.CompletedProcess:
        wrap_exec = [sys.executable, Path(__file__).parent / "wrap.py"]
        try:
            return subprocess.run(
                wrap_exec + args,
                capture_output=True,
                universal_newlines=True,
            )
        except subprocess.CalledProcessError as e:
            print(e.stdout)
            print(e.stderr)
            raise


class TestUnknownFunctionHandling(WrapTestCase):
    def test_unknown_mpi_function(self):
        content = """{{fn FOO MPI_unknown}}
  	// Do-nothing wrapper for {{FOO}}
{{endfn}}
"""
        with TmpFile(content) as wrapper:
            proc = self.exec_wrap([wrapper])
            self.assertNotEqual(proc.returncode, 0)
            self.assertIn("MPI_unknown is not an MPI function", proc.stderr)

    def test_ignore_unknown_mpi_function(self):
        content = """{{fn FOO MPI_unknown}}
  	// Do-nothing wrapper for {{FOO}}
{{endfn}}
"""
        with TmpFile(content) as wrapper:
            proc = self.exec_wrap(["-n", wrapper])
            self.assertEqual(proc.returncode, 0)
            self.assertNotIn("MPI_unknown is not an MPI function", proc.stderr)


if __name__ == "__main__":
    unittest.main()
