/* This file is part of GTI (Generic Tool Infrastructure)
 *
 * Copyright (C)
 *  2008-2019 ZIH, Technische Universitaet Dresden, Federal Republic of Germany
 *  2008-2019 Lawrence Livermore National Laboratories, United States of America
 *  2013-2019 RWTH Aachen University, Federal Republic of Germany
 *
 * See the LICENSE file in the package base directory for details
 */

/**
 * @file CProtMpiSplitModule.cpp generated from cprot_mpi_split_module.w
 *       Splits MPI processes into multiple sets of processes.
 *       Intention is to use one set(id:0) for the actual application
 *       and the remaining sets for tool processes (one set for
 *       each level of the tool).
 *       Further, a separate stack is used for each of the sets,
 *       to enable tools with distinct layouts. Also, the actual
 *       application calls are separated such that MPI_COMM_WORLD
 *       is replaced by a comm representing the application
 *       processes set.
 *       For all the tool process sets only an MPI_Init and an
 *       MPI_Finalize is called, afterwards an exit(0) is issued,
 *       so all tool startup should be done in MPI_Init and it should
 *       only return once a shutdown is desired.
 *
 * @author Tobias Hilbrich
 * @date 27.07.2009
 */

#include <assert.h>
#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <sys/time.h>
#include <vector>
#include <map>
#include <string>
#include <unistd.h>
#include <sstream>
#include <sstream>
#include <cstring>
#include <dlfcn.h>

#include <pnmpimod.h>
#include "GtiMacros.h"
#include "SplitProcessesModuleProxy.h"

#include "gtiConfig.h"

/**
 * Global definitions and helpers.
 */
/*Name of split module @todo move to some common place, currently present twice*/
#define MUST_SPLIT_MODULE_NAME "replace_module"
MPI_Comm fakeComm;
#define MACRO_MPI_Comm(_c)                                                                         \
    {                                                                                              \
        if (_c == MPI_COMM_WORLD)                                                                  \
            _c = fakeComm;                                                                         \
    }

static PNMPI_modHandle_t stack{0};

/**
 * PNMPI_ReistrationPoint
 */
extern "C" void PNMPI_RegistrationPoint()
{
    /* register this module*/
#if !defined(NDEBUG)
    int err =
#endif
        PNMPI_Service_RegisterModule(MUST_SPLIT_MODULE_NAME);
    assert(err == PNMPI_SUCCESS);

#if !defined(NDEBUG)
    err =
#endif
        PNMPI_Service_GetStackByName("level_0", &stack);
    assert(err == PNMPI_SUCCESS);
}

#ifdef MUST_TIME
struct timeval gStart, gEnd;
#endif

#if defined(GTI_ENABLE_SESSIONS)
/**
 * Remove tool ranks from a given group.
 * The caller is responsible to call MPI_Group_free() on the returned group.
 *
 * @param group the input group
 * @return the intersection of the group with the ranks of this layer
 */
static MPI_Group sanitizedGroup(MPI_Group group)
{
    MPI_Group mySetGroup = MPI_GROUP_NULL;
    XMPI_Comm_group_NewStack(::stack, gti::SplitProcessesModuleProxy().getMySetComm(), &mySetGroup);

    MPI_Group sanitizedGroup = MPI_GROUP_NULL;
    XMPI_Group_intersection_NewStack(::stack, mySetGroup, group, &sanitizedGroup);

    XMPI_Group_free_NewStack(::stack, &mySetGroup);
    return sanitizedGroup;
}

/**
 * Wrapper that ensures that ranks internal to GTI are removed from the requested process sets.
 */
extern "C" int
MPI_Group_from_session_pset(MPI_Session session, const char* pset_name, MPI_Group* newgroup)
{
    MPI_Group realGroup = MPI_GROUP_NULL;
    int err = XMPI_Group_from_session_pset(session, pset_name, &realGroup);
    assert(err == MPI_SUCCESS);

    *newgroup = sanitizedGroup(realGroup);

    XMPI_Group_free_NewStack(::stack, &realGroup);
    return err;
}

/**
 * Wrapper that subtracts  the tool processes from the value of `mpi_size` for the process set.
 */
extern "C" int MPI_Session_get_pset_info(MPI_Session session, const char* pset_name, MPI_Info* info)
{
    // Actual library call
    int const ret = XMPI_Session_get_pset_info(session, pset_name, info);

    // Find out the size of the group excluding tool ranks
    MPI_Group fullGroup = MPI_GROUP_NULL;
    XMPI_Group_from_session_pset_NewStack(::stack, session, pset_name, &fullGroup);
    MPI_Group sanitized_group = sanitizedGroup(fullGroup);
    int size = -1;
    XMPI_Group_size_NewStack(::stack, sanitized_group, &size);

    // Replace the size in the info object
    XMPI_Info_set_NewStack(::stack, *info, "mpi_size", std::to_string(size).c_str());

    // Cleanup and return!
    XMPI_Group_free_NewStack(::stack, &sanitized_group);
    XMPI_Group_free_NewStack(::stack, &fullGroup);
    return ret;
}
#endif

#include "CProtMpiRenameWorld.wrap.cpp" // NOLINT(bugprone-suspicious-include)
