/* This file is part of GTI (Generic Tool Infrastructure)
 *
 * Copyright (C)
 *  2008-2019 ZIH, Technische Universitaet Dresden, Federal Republic of Germany
 *  2008-2019 Lawrence Livermore National Laboratories, United States of America
 *  2013-2019 RWTH Aachen University, Federal Republic of Germany
 *
 * See the LICENSE file in the package base directory for details
 */

/**
 * @file CProtMpiSplitModule.cpp generated from cprot_mpi_split_module.w
 *       Splits MPI processes into multiple sets of processes.
 *       Intention is to use one set(id:0) for the actual application
 *       and the remaining sets for tool processes (one set for
 *       each level of the tool).
 *       Further, a separate stack is used for each of the sets,
 *       to enable tools with distinct layouts. Also, the actual
 *       application calls are separated such that MPI_COMM_WORLD
 *       is replaced by a comm representing the application
 *       processes set.
 *       For all the tool process sets only an MPI_Init and an
 *       MPI_Finalize is called, afterwards an exit(0) is issued,
 *       so all tool startup should be done in MPI_Init and it should
 *       only return once a shutdown is desired.
 *
 * @author Tobias Hilbrich
 * @date 27.07.2009
 */

#include "CProtMpiSplitWorld.h"

#include <algorithm>
#include <assert.h>
#include <cstddef>
#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <iomanip>
#include <sys/time.h>
#include <sys/resource.h>
#include <vector>
#include <map>
#include <string>
#include <unistd.h>
#include <sstream>

#include <dlfcn.h>

#include <pnmpimod.h>
#include "GtiMacros.h"
#include "SplitProcessesModuleProxy.h"

#include "gtiConfig.h"
#include "CStratCrashHandling.h"

/**
 * Global definitions and helpers.
 */
/*Name of split module @todo move to some common place, currently present twice*/
#define MUST_SPLIT_MODULE_NAME "split_processes"

struct process_set_t {
    int size;
    int start_rank;
    int in_set : 2;    ///< If current rank is part of this set.
    int mpi_place : 2; ///< If this set consists of MPI places.
    int app_place : 2; ///< If this set consists of application places.
    MPI_Comm set_comm;
    std::size_t set_index;
    PNMPI_modHandle_t stack;
};

static std::vector<process_set_t> g_sets;
static std::map<std::pair<int, int>, int>
    g_mappings; /**< Maps a (OwnSetIndex,CommID) pair to a SetIndex, usage is to match equal
                   comm_ids to the right set indices.*/

#if defined(GTI_ENABLE_SESSIONS)
/** Handle for the GTI internal MPI Session. */
static MPI_Session g_session = MPI_SESSION_NULL;
#endif

/**
 * The below global variable is used for the MUST-DDT integration.
 * It has the following meaning:
 * # -1 => this is a tool process
 * # 0-N => this is application rank i
 */
// static int MUST_application_rank = -1;

static bool gPrintMemoryConsumption = 0;

/**
 * This function serves for debugger integrations:
 * MUST issues it when its own initialization is finished to tell a potentially attached debugger
 * that the MUST initialization as well as MPI_Init was performed.
 */
extern "C" __attribute__((noinline)) void MUST_InitComplete(void)
{
    asm("");
#ifdef MUST_DEBUG
    std::cout << "MUST: " << getpid() << " has completed initialization." << std::endl;
#endif
}

/**
 * Query functions.
 */

/**
 * Getter for the process set that the current rank is in.
 * @return reference to the process set
 */
static const process_set_t& getMyMpiSet()
{
    const auto isMyMpiSet = [](const process_set_t& set) -> bool {
        return set.mpi_place && set.in_set == 1;
    };
    const auto mySetIt = std::find_if(g_sets.cbegin(), g_sets.cend(), isMyMpiSet);
    assert(mySetIt != g_sets.cend());
    return *mySetIt;
}

extern "C" int getMySetSize(int* size)
{
    *size = -1;

    for (const auto& set : g_sets) {
        if (set.in_set == 1)
            *size = set.size;
    }
    return PNMPI_SUCCESS;
}

/** Flag that tells if the World Model is used. */
static bool usingWorldModel = false;
/** Handle to a full world communicator. */
static MPI_Comm realCommWorld = MPI_COMM_NULL;
/** Handle to a full self communicator. */
static MPI_Comm g_realCommSelf = MPI_COMM_NULL;

static PNMPI_modHandle_t stack{0};

extern "C" int getRealCommWorld(void* comm)
{
    *((MPI_Comm*)comm) = realCommWorld;
    return PNMPI_SUCCESS;
}

extern "C" int getMySetComm(MPI_Comm* comm)
{
    *((MPI_Comm*)comm) = MPI_COMM_NULL;

    for (const auto& set : g_sets) {
        if (set.in_set == 1)
            *((MPI_Comm*)comm) = set.set_comm;
    }
    return PNMPI_SUCCESS;
}

extern "C" int getSetInfo(int commId, int* set_size, int* start_rank)
{
    int ownSetId = 0;
    unsigned setIdToUse;

    // get own set id
    for (std::size_t i = 0; i < g_sets.size(); i++) {
        if (g_sets[i].in_set == 1)
            ownSetId = i;
    }

    // find set id to use
    std::map<std::pair<int, int>, int>::iterator iter =
        g_mappings.find(std::make_pair(ownSetId, commId));

    if (iter != g_mappings.end())
        setIdToUse = iter->second; // New version use the mapping (if given)
    else
        setIdToUse = commId; // Old version gicen commId is the setId to use

    assert(setIdToUse < g_sets.size() && setIdToUse >= 0);

    *set_size = g_sets[setIdToUse].size;
    *start_rank = g_sets[setIdToUse].start_rank;

    return PNMPI_SUCCESS;
}

/**
 * PNMPI_ReistrationPoint
 */
extern "C" void PNMPI_RegistrationPoint()
{
    /* register this module*/
#if !defined(NDEBUG)
    int err =
#endif
        PNMPI_Service_RegisterModule(MUST_SPLIT_MODULE_NAME);
    assert(err == PNMPI_SUCCESS);

#if !defined(NDEBUG)
    err =
#endif
        PNMPI_Service_GetStackByName("level_0", &stack);
    assert(err == PNMPI_SUCCESS);
}

#ifdef MUST_TIME
static struct timeval gStart, gEnd;
#endif

void printSetMemoryConsumption(int gset_index)
{
    struct rusage r;
    getrusage(RUSAGE_SELF, &r);

    MPI_Comm setComm = g_sets[gset_index].set_comm;

    int commRank, commSize;
    XMPI_Comm_rank_NewStack(::stack, setComm, &commRank);
    XMPI_Comm_size_NewStack(::stack, setComm, &commSize);

    if (commRank != 0) {
        XMPI_Gather_NewStack(::stack, &r.ru_maxrss, 1, MPI_LONG, nullptr, 1, MPI_LONG, 0, setComm);
    } else {
        // Collect and sum up data on a single rank for ordered output
        long procMemory[commSize];
        long totalMemory = 0;

        XMPI_Gather_NewStack(
            ::stack,
            &r.ru_maxrss,
            1,
            MPI_LONG,
            procMemory,
            1,
            MPI_LONG,
            0,
            setComm);

        for (int j = 0; j < commSize; ++j) {
            totalMemory += procMemory[j];
        }

        std::stringstream memory_msg;
        memory_msg << std::setprecision(3) << std::fixed;
        memory_msg << "[GTI] Total Memory Consumption Layer " << gset_index
                   << " (max_rss): " << totalMemory / 1024.0 << " MiB" << std::endl;
        for (int j = 0; j < commSize; ++j) {
            memory_msg << "[GTI] Rank " << j << ", Layer " << gset_index << ":\t"
                       << procMemory[j] / 1024.0 << " MiB " << std::endl;
        }

        std::cout << memory_msg.str();
    }
}

/**
 * Determine if this process is in the top layer (i.e. the root of the TBON).
 *
 * @return true if this process is in the top layer
 */
static bool isTopLayer() { return g_sets.size() - 1 == getMyMpiSet().set_index; }

extern "C" int MPI_Finalize()
{
    if (gPrintMemoryConsumption) {
        if (g_sets[0].app_place) {
            printSetMemoryConsumption(0);
        } else {
            // If we run MUST in hybrid mode, we output the result of
            // layer 1 here (which represents thread_app + thread_place layers).
            printSetMemoryConsumption(1);
        }
    }

    if (::usingWorldModel) {
        if (::realCommWorld != MPI_COMM_NULL) {
            XMPI_Comm_free_NewStack(::stack, &::realCommWorld);
        }
        if (::g_realCommSelf != MPI_COMM_NULL) {
            XMPI_Comm_free_NewStack(::stack, &::g_realCommSelf);
        }
    }

    return XMPI_Finalize();
}

/**
 * Send the finalize event to all modules in the stack.
 *
 * @param stack the stack to send the finalize event to
 */
static void finalizeTool(PNMPI_modHandle_t stack)
{
#if !defined(NDEBUG)
    int const err =
#endif
        XMPI_Finalize_NewStack(stack);
    assert(err == MPI_SUCCESS);
}

/**
 * Execute the current rank as a tool process.
 *
 * @param stack the stack of tool modules
 * @param argc argument count
 * @param argv argument array
 * @param required required mpi thread level
 * @param provided[out] threadlevel provided by the MPI implementation
 */
static void runTool(PNMPI_modHandle_t stack, int* argc, char*** argv, int required, int* provided)
{
    /*Init the MPI tool place, when it returns finalize and exit*/
#if !defined(NDEBUG)
    int const err =
#endif
        XMPI_Init_thread_NewStack(stack, argc, argv, required, provided);
    assert(err == MPI_SUCCESS);

    if (gPrintMemoryConsumption)
        printSetMemoryConsumption(std::distance(&*g_sets.cbegin(), &(getMyMpiSet())));

#ifdef MUST_TIME
    gettimeofday(&gEnd, NULL);

    if (rank == world_size - 1)
        std::cout << "Time Post Init - Pre Finalize of Tool " << rank << " (usec): "
                  << ((gEnd.tv_sec * 1000000 + gEnd.tv_usec) -
                      (gStart.tv_sec * 1000000 + gStart.tv_usec))
                  << std::endl;
#endif

    /*
     * This does not calls the real MPI_Finalize,
     * it just passes the finalize to all modules of
     * the stack, we must still call the actual finalize
     * afterwards.
     */
    finalizeTool(stack);
    exit(0);
}

/**
 * Execute the current rank as a tool process.
 *
 * @param stack the stack of tool modules
 * @param argc argument count
 * @param argv argument array
 */
static void runTool(PNMPI_modHandle_t stack, int* argc, char*** argv)
{
    runTool(stack, argc, argv, MPI_THREAD_SINGLE, nullptr);
}

/**
 * Service function that starts the tool stack.
 */
extern "C" int gtiRunTool()
{
    int provided = -1;
    auto const toolStack = getMyMpiSet().stack;

    runTool(toolStack, nullptr, nullptr, MPI_THREAD_MULTIPLE, &provided);
    return PNMPI_FAILURE; // This should not be possible!
}

#if defined(GTI_ENABLE_SESSIONS)
static void init(MPI_Comm commWorld, MPI_Comm commSelf);

/**
 * Helper that creates the communicator for a given process set.
 * @param session the MPI Session to derive the comm from
 * @param process_set process set uri, e.g. "mpi://WORLD"
 * @return the newly created communicator
 */
static MPI_Comm getCommForProcessSet(MPI_Session session, const char* process_set)
{
    assert(session != MPI_SESSION_NULL);
#if !defined(NDEBUG)
    int err = MPI_SUCCESS;
#endif

    MPI_Group group = MPI_GROUP_NULL;
#if !defined(NDEBUG)
    err =
#endif
        XMPI_Group_from_session_pset_NewStack(::stack, session, process_set, &group);
    assert(err == MPI_SUCCESS);
    assert(group != MPI_GROUP_NULL);

    MPI_Comm comm = MPI_COMM_NULL;
#if !defined(NDEBUG)
    err =
#endif
        XMPI_Comm_create_from_group_NewStack(
            ::stack,
            group,
            "gti",
            MPI_INFO_NULL,
            gti::getCommErrhandler(),
            &comm);
    assert(err == MPI_SUCCESS);
    assert(comm != MPI_COMM_NULL);

    XMPI_Group_free_NewStack(::stack, &group);
    return comm;
}

/**
 * Library constructor that initializes the internal session.
 */
__attribute__((constructor)) static void GTI_Init()
{
    // We _must not_ use the XMPI_xxx_NewStack functions here. PnMPI might not be initialized yet.
    MPI_Info info = MPI_INFO_NULL;
    PMPI_Info_create(&info);
    PMPI_Info_set(info, "thread_level", "MPI_THREAD_MULTIPLE");

    gti::crashHandlingInit();
    PMPI_Session_init(info, MPI_ERRORS_ARE_FATAL, &::g_session);
    PMPI_Session_set_errhandler(::g_session, gti::getSessionErrhandler());

    PMPI_Info_free(&info);
}

/**
 * Library destructor that frees the internal session.
 */
__attribute__((destructor)) static void GTI_Fini()
{
    if (!::usingWorldModel) {
        if (::realCommWorld != MPI_COMM_NULL) {
            XMPI_Comm_free_NewStack(::stack, &::realCommWorld);
        }
        if (::g_realCommSelf != MPI_COMM_NULL) {
            XMPI_Comm_free_NewStack(::stack, &::g_realCommSelf);
        }
    }

    if (g_session != MPI_SESSION_NULL) {
        PMPI_Session_finalize(&::g_session);
    }
}

void PNMPI_Fini(void) { GTI_Fini(); }

/**
 * Hook called by PnMPI after all modules have been registered.
 */
extern "C" void PNMPI_Init()
{
    gti::crashHandlingInitErrhandlers();

    auto commWorld = getCommForProcessSet(::g_session, "mpi://WORLD");
    auto commSelf = getCommForProcessSet(::g_session, "mpi://SELF");

    init(commWorld, commSelf);
}
#endif

/**
 * Flag that indicates that this module has been already initialized.
 */
static int gModuleInitialized = 0;

/** Initialize the module.
 *
 * Takes ownership of the passed communicators.
 *
 * @param commWorld a communicator corresponding to MPI_COMM_WORLD
 * @param commSelf a communicator corresponding to MPI_COMM_SELF
 */
static void init(MPI_Comm commWorld, MPI_Comm commSelf)
{
    int err;
    const char* inp;
    int split_mod;
    int num_sets, nodesize = 0;
    int world_size, rank;
    int curr_set_start = 0;
    PNMPI_Service_descriptor_t service;

    ::realCommWorld = commWorld;
    ::g_realCommSelf = commSelf;

    err = XMPI_Comm_size_NewStack(::stack, commWorld, &world_size);
    assert(err == MPI_SUCCESS);
    err = XMPI_Comm_rank_NewStack(::stack, commWorld, &rank);
    assert(err == MPI_SUCCESS);

    char buf[512];
    PNMPI_modHandle_t stack = 0;

    err = PNMPI_Service_GetModuleSelf(&split_mod);
    assert(err == PNMPI_SUCCESS);

    sprintf(buf, "stack_0");
    err = PNMPI_Service_GetArgument(split_mod, buf, &inp);
    assert(err == PNMPI_SUCCESS);
    err = PNMPI_Service_GetStackByName("level_0", &stack);
    assert(err == PNMPI_SUCCESS);

#ifdef MUST_TIME
    gettimeofday(&gStart, NULL);
#endif

    // enable a sleep to allow attaching with a debugger
    if (getenv("MUST_WAIT_AT_STARTUP") != NULL) {
        //        std::cout << "Rank " << rank << " has pid " <<
        //        getpid() << std::endl;
        // Stop random interleaving when printing
        printf("Rank %i has pid %i\n", rank, getpid());
        sleep(atoi(getenv("MUST_WAIT_AT_STARTUP")));
    }

    // print out memory statistics if requested
    if ((getenv("GTI_VERBOSE") != NULL && atoi(getenv("GTI_VERBOSE")) > 0) ||
        (getenv("INTERNAL_GTI_PRINT_MEMORY_CONSUMPTION") != NULL &&
         atoi(getenv("INTERNAL_GTI_PRINT_MEMORY_CONSUMPTION")) == 1)) {
        gPrintMemoryConsumption = true;
    }

    /*
     * EXPERIMENTAL, might be of use in the future.
     * Idea is to automatically connect a gdbserver to some ranks to allow remote
     * debugging in unfriendly environments.
     */
    if (getenv("MUST_START_GDBSERVER_BEGIN") != NULL &&
        getenv("MUST_START_GDBSERVER_END") != NULL) {
        int begin = atoi(getenv("MUST_START_GDBSERVER_BEGIN"));
        int end = atoi(getenv("MUST_START_GDBSERVER_END"));

        if (rank >= begin && rank <= end) {
            char host[512];
            gethostname(host, 512);
            std::stringstream stream;
            stream << "gdbserver " << host << ":" << rank + 10000 << " --attach " << getpid()
                   << " &" << std::endl;
            if (system(stream.str().c_str()) == 0)
                std::cout << "GDBSERVER for rank " << rank << " listens on " << host << ":"
                          << rank + 10000 << std::endl;
            else
                std::cout << "Starting GDBSERVER for rank " << rank << " failed!" << std::endl
                          << "Command was: " << stream.str() << std::endl;
        }
    }

    /* Query for nodesize */
    err = PNMPI_Service_GetArgument(split_mod, "nodesize", &inp);
    if (err == PNMPI_SUCCESS)
        nodesize = atoi(inp);

    /* Query for number of sets and their stacks */
    err = PNMPI_Service_GetArgument(split_mod, "num_sets", &inp);
    assert(err == PNMPI_SUCCESS);
    num_sets = atoi(inp);
    g_sets.resize(num_sets);

    /*   int in_sets = 0; */
    int found_app = 0;
    for (int i = 0; i < num_sets; i++) {
        int size;

        sprintf(buf, "size_%d", i);
        err = PNMPI_Service_GetArgument(split_mod, buf, &inp);
        assert(err == PNMPI_SUCCESS);
        size = atoi(inp);

        if (i > 0) {
            sprintf(buf, "stack_%d", i);
            err = PNMPI_Service_GetArgument(split_mod, buf, &inp);
            assert(err == PNMPI_SUCCESS);
            err = PNMPI_Service_GetStackByName(inp, &stack);
            assert(err == PNMPI_SUCCESS);
        }

        /* Get the place for the set. */
        sprintf(buf, "place_%d", i);
        inp = PNMPI_Service_GetArgumentSelf(buf);
        assert(inp != NULL);
        g_sets[i].app_place = 0;
        if (strncmp("mpi_place", inp, 10) == 0) {
            g_sets[i].mpi_place = 1;
            if (!found_app) {
                g_sets[i].app_place = 1;
                found_app = 1;
            }
        } else {
            g_sets[i].mpi_place = 0;
            g_sets[i].set_comm = commSelf;
            continue;
        }

        if (size > 0)
            g_sets[i].start_rank = curr_set_start;
        else
            g_sets[i].start_rank = g_sets[0].start_rank;

        int last_rank = curr_set_start + size - 1;
        g_sets[i].in_set = 0;
        g_sets[i].set_index = i;
        g_sets[i].stack = stack;

        if (nodesize > 0 && i < 2 &&
            num_sets > 1) { /* for given nodesize, distribute the lower levels:
                               1 tool rank + nodesize-1 app ranks per node */
            if (i == 0) {
                g_sets[0].start_rank = 1;
                g_sets[0].size = size;
                /* one tool node per computenode */
                /* \lfloor size / (nodesize-1) \rfloor  */
                g_sets[1].size = (size - 1) / (nodesize - 1) + 1;
                last_rank = g_sets[0].size + g_sets[1].size;
                if (rank < last_rank && (rank % nodesize) != 0 && size > 0)
                    g_sets[i].in_set = 1;
            } else /* i==1 */
            {
                g_sets[1].start_rank = 0;
                assert(g_sets[i].size == size);
                if (rank < last_rank && (rank % nodesize) == 0)
                    g_sets[i].in_set = 1;
            }
        } else /* old split behaviour for nodesize = 0 or higher levels
                */
        {
            g_sets[i].size = size;
            if (rank >= g_sets[i].start_rank && rank <= last_rank && size > 0)
                g_sets[i].in_set = 1;
        }

        err = XMPI_Comm_split_NewStack(stack, commWorld, g_sets[i].in_set, 0, &g_sets[i].set_comm);
        assert(err == MPI_SUCCESS);

        /*        // MUST-DDT Integration Begin
                if (g_sets[i].in_set) {
                    if (i == 0) {
                        int myNewAppRank;
                        PMPI_Comm_rank(g_sets[i].set_comm, &myNewAppRank);
                        MUST_application_rank = myNewAppRank;
                    } else {
                        MUST_application_rank = -1;
                    }
                }
                // MUST-DDT Integration END*/

        curr_set_start += size;
        assert(world_size >= curr_set_start); // Sets must require no more processes
                                              // than available!
    }
    /* Query for mappings */
    err = PNMPI_Service_GetArgument(split_mod, "num_mappings", &inp);
    if (err == PNMPI_SUCCESS) {
        int numMappings = atoi(inp);

        for (int i = 0; i < numMappings; i++) {
            char buf[512];

            sprintf(buf, "mapping%d", i);
            err = PNMPI_Service_GetArgument(split_mod, buf, &inp);
            assert(err == PNMPI_SUCCESS);

            std::string mapping(inp);

            int ownSetId, commId, setIdToUse;
            std::string ownSetIdStr, commIdStr, setIdToUseStr;

            // Own set id
            size_t pos = mapping.find_first_of(':');
            assert(pos != std::string::npos);
            ownSetIdStr = mapping.substr(0, pos);

            // comm id
            size_t pos2 = mapping.find_first_of(':', pos + 1);
            assert(pos2 != std::string::npos);
            commIdStr = mapping.substr(pos + 1, pos2 - pos - 1);

            // Set id to use
            setIdToUseStr = mapping.substr(pos2 + 1, mapping.length() - pos2 - 1);

            ownSetId = atoi(ownSetIdStr.c_str());
            commId = atoi(commIdStr.c_str());
            setIdToUse = atoi(setIdToUseStr.c_str());

            g_mappings.insert(std::make_pair(std::make_pair(ownSetId, commId), setIdToUse));
        }
    }

    /* Provide services to query for set information */
    sprintf(service.name, "SplitMod_getMySetSize");
    service.fct = (PNMPI_Service_Fct_t)getMySetSize;
    sprintf(service.sig, "p");
    err = PNMPI_Service_RegisterService(&service);
    assert(err == PNMPI_SUCCESS);

    sprintf(service.name, "SplitMod_getMySetComm");
    service.fct = (PNMPI_Service_Fct_t)getMySetComm;
    sprintf(service.sig, "p");
    err = PNMPI_Service_RegisterService(&service);
    assert(err == PNMPI_SUCCESS);

    sprintf(service.name, "SplitMod_getRealCommWorld");
    service.fct = (PNMPI_Service_Fct_t)getRealCommWorld;
    sprintf(service.sig, "p");
    err = PNMPI_Service_RegisterService(&service);
    assert(err == PNMPI_SUCCESS);

    sprintf(service.name, "SplitMod_getSetInfo");
    service.fct = (PNMPI_Service_Fct_t)getSetInfo;
    sprintf(service.sig, "ipp");
    err = PNMPI_Service_RegisterService(&service);
    assert(err == PNMPI_SUCCESS);

    sprintf(service.name, "SplitMod_runTool");
    service.fct = (PNMPI_Service_Fct_t)gtiRunTool;
    sprintf(service.sig, "v");
    err = PNMPI_Service_RegisterService(&service);
    assert(err == PNMPI_SUCCESS);

    // TODO: Add service function for registering a panic callback to. It forwards to the crash
    //  handling code and gets called when a panic is to be raised by the error handler.
    sprintf(service.name, "SplitMod_onPanic");
    service.fct = (PNMPI_Service_Fct_t)gti::onPanic;
    sprintf(service.sig, "p");
    err = PNMPI_Service_RegisterService(&service);
    assert(err == PNMPI_SUCCESS);

    {
        // Tell the crashhandler of mySetComm
        if (!isTopLayer()) {
            // Do not attach for top layer to preserve to old behaviour from adding the crash
            // handling source to every up comm strategy.
            MPI_Comm mySetComm = MPI_COMM_NULL;
            getMySetComm(&mySetComm);
            gti::attachErrorhandler(mySetComm);
        }
    }
    gti::registerPredefinedErrhandlers();

    /* Call MPI_Init for the correct stack */
    MUST_InitComplete(); // We are all set up and ready when now; MPI_Init was
                         // invoked and we decided who is application and who is
                         // tool
    gModuleInitialized = 1;
}

#include "CProtMpiSplitWorld.wrap.cpp" // NOLINT(bugprone-suspicious-include)
