/*
 * Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file CStratCrashHandling.h
 *
 *  @date 03.10.23
 *  @author Sebastian Grabowski
 */

#ifndef MUST_EXTERNALS_GTI_MODULES_COMM_STRATEGY_CSTRATCRASHHANDLING_H
#define MUST_EXTERNALS_GTI_MODULES_COMM_STRATEGY_CSTRATCRASHHANDLING_H

#include <mpi.h>

namespace gti
{
/**
 * Initialize the cStratCrashHandling module.
 *
 * This initializes all variables except those that require an initialized MPI.
 */
auto crashHandlingInit() -> void;

/**
 * Initialize the MPI errorhandlers of this module.
 *
 * This must be called with an initialized MPI.
 */
auto crashHandlingInitErrhandlers() -> void;

auto crashHandlingPnMPIReady() -> void;

/**
 * Initialize the MPI world model errorhandlers of this module.
 *
 * This must be called with an initialized MPI World Model!
 */
auto crashHandlingInitWorldModel() -> void;

/**
 * Register the predefined errorhandlers with the errorhandler tracker.
 *
 * \note _Must_ be called outside library constructor. Otherwise the locks used might not be
 * initialized!
 */
auto registerPredefinedErrhandlers() -> void;

/**
 * Get the Session errorhandler if this module.
 *
 * Requires the error handlers to be initialized (@see crashHandlingInitErrhandlers()).
 *
 * @return the Session errorhandler
 */
auto getSessionErrhandler() -> MPI_Errhandler;

/**
 * Get the Comm errorhandler if this module.
 *
 * Requires the error handlers to be initialized (@see crashHandlingInitErrhandlers()).
 *
 * @return the Comm errorhandler
 */
auto getCommErrhandler() -> MPI_Errhandler;

/**
 * Attach the Comm errorhandler of this module.
 *
 * @param comm The communicator to attach the errorhandler to
 */
auto attachErrorhandler(MPI_Comm comm) -> void;

/** Signature of the panic callback functions. */
using PanicCallbackType = void();

/**
 * Register callbacks to be invoked when packicking.
 *
 * @param callback the function to call on a panic
 */
auto onPanic(PanicCallbackType* callback) -> void;

} // namespace gti

#endif // MUST_EXTERNALS_GTI_MODULES_COMM_STRATEGY_CSTRATCRASHHANDLING_H
