/* This file is part of GTI (Generic Tool Infrastructure)
 *
 * Copyright (C)
 *  2008-2019 ZIH, Technische Universitaet Dresden, Federal Republic of Germany
 *  2008-2019 Lawrence Livermore National Laboratories, United States of America
 *  2013-2023 RWTH Aachen University, Federal Republic of Germany
 *
 * See the LICENSE file in the package base directory for details
 */

/**
 * @file SplitProcessesModuleProxy.h
 *       Header for the MPI split module.
 *
 * @author Sebastian Grabowski
 */

#ifndef SPLIT_PROCESSES_MODULE_PROXY_H_
#define SPLIT_PROCESSES_MODULE_PROXY_H_

#include <stdexcept>

#include <pnmpi/service.h>
#include <pnmpimod.h>

namespace gti
{

/** Indicates that PnMPI could not find a requested module. */
class ModuleNotFoundError : public std::runtime_error
{
  public:
    ModuleNotFoundError(const char* modname)
        : std::runtime_error(std::string("Module ") + modname + " is not loaded.")
    {
    }
};

/** Indicates that PnMPI could not find a service of the module. */
class ServiceNotFoundError : public std::runtime_error
{
  public:
    ServiceNotFoundError(const char* srvname)
        : std::runtime_error(std::string("Service ") + srvname + " was not found.")
    {
    }
};

/** Indicates that the given signature did not match to the registered one. */
class SignatureNotFound : public std::runtime_error
{
  public:
    SignatureNotFound(const char* srvname, const char* signature)
        : std::runtime_error(
              std::string("The signature of the service function \"") + srvname +
              "\" does not match \"" + signature + "\".")
    {
    }
};

/** Indicates some PnMPI error. */
class PnMPIError : public std::runtime_error
{
    /** The cause as returned by PnMPI function calls. */
    pnmpi_status myStatus;

  public:
    PnMPIError(pnmpi_status status) : std::runtime_error(""), myStatus{status} {}
};

/**
 * C++-style proxy for the service functions of the "split_processes" module.
 *
 * It provides a thin convenience wrapper around the rather verbose PnMPI services api.
 */
class SplitProcessesModuleProxy
{
    /** The name of the actual module as registered at PnMPI. */
    constexpr static const char* const myName = "split_processes";

    /** The handle to the module */
    PNMPI_modHandle_t myHandle = PNMPI_MODHANDLE_NULL;

    /** The type of the function for `SplitMod_getMySetComm`. */
    using GetMySetCommType = int(MPI_Comm*);
    /** The function pointer to `SplitMod_getMySetComm`. */
    GetMySetCommType* myGetMySetComm = nullptr;

    /** The type of the function for `SplitMod_getRealCommWorld`. */
    using GetRealCommWorldType = int(MPI_Comm*);
    /** The function pointer to `SplitMod_getRealCommWorld`. */
    GetRealCommWorldType* myGetRealCommWorld = nullptr;

    /** The type of the function for `SplitMod_runTool`. */
    using MyRunToolType = int();
    /** The function pointer to `SplitMod_runTool`. */
    MyRunToolType* myRunTool = nullptr;

    /** The type of the function for `SplitMod_onPanic`. */
    using MyOnPanicType = void(void (*)());
    /** The function pointer to `SplitMod_onPanic`. */
    MyOnPanicType* myOnPanic = nullptr;

    /**
     * Get the PnMPI service function by its registered name.
     *
     * @tparam FunctionType the type of the service function
     * @param name the registered name of the service function
     * @return the function pointer of the service function
     *
     * @throws ServiceNotFoundError when the PnMPI module could not resolve the service name
     * @throws SignatureNotFound when the given signature does not match the registered one
     * @throws PnMPIError otherwise
     */
    template <class FunctionType>
    auto getServiceFunction(const char* name, const char* signature) const -> FunctionType*
    {
        static_assert(
            std::is_function<FunctionType>::value,
            "Given template argument is not a function type");

        PNMPI_Service_descriptor_t service;
        auto const pnmpi_err = PNMPI_Service_GetServiceByName(myHandle, name, signature, &service);
        if (pnmpi_err != PNMPI_SUCCESS) {
            switch (pnmpi_err) {
            case PNMPI_NOSERVICE:
                throw ServiceNotFoundError{name};
            case PNMPI_SIGNATURE:
                throw SignatureNotFound{name, signature};
            default:
                throw PnMPIError{pnmpi_err};
            }
        }
        return reinterpret_cast<FunctionType*>(service.fct);
    };

  public:
    /**
     * Constructor.
     *
     * \throws ModuleNotFoundError when the PnMPI module does not exist
     * \throws PnMPIError otherwise
     */
    SplitProcessesModuleProxy()
    {
        auto const pnmpi_err = PNMPI_Service_GetModuleByName("split_processes", &myHandle);
        if (pnmpi_err != PNMPI_SUCCESS) {
            switch (pnmpi_err) {
            case PNMPI_NOMODULE:
                throw ModuleNotFoundError{myName};
            default:
                throw PnMPIError{pnmpi_err};
            }
        }

        myGetMySetComm = getServiceFunction<GetMySetCommType>("SplitMod_getMySetComm", "p");
        myGetRealCommWorld =
            getServiceFunction<GetRealCommWorldType>("SplitMod_getRealCommWorld", "p");
        myRunTool = getServiceFunction<MyRunToolType>("SplitMod_runTool", "v");
        myOnPanic = getServiceFunction<MyOnPanicType>("SplitMod_onPanic", "p");
    }

    /**
     * Proxy function for the "SplitMod_getMySetComm" service function.
     *
     * @see ::getMySetComm(void*)
     */
    MPI_Comm getMySetComm() const
    {
        MPI_Comm comm = MPI_COMM_NULL;
        myGetMySetComm(&comm);
        return comm;
    }

    /**
     * Proxy function for the "SplitMod_getRealCommWorld" service function.
     *
     * @see ::getRealCommWorld(void*)
     */
    MPI_Comm getRealCommWorld() const
    {
        MPI_Comm comm = MPI_COMM_NULL;
        myGetRealCommWorld(&comm);
        return comm;
    }

    /**
     * Proxy function for the "SplitMod_runTool" service function.
     *
     * @see ::gtiRunTool()
     */
    void runTool() const { myRunTool(); }

    /**
     * Proxy function for the "SplitMod_onPanic" service function.
     *
     * @see gti::onPanic(void (*)())
     */
    void onPanic(void (*callback)()) { myOnPanic(callback); }
};

} // namespace gti

#endif /* SPLIT_PROCESSES_MODULE_PROXY_H_ */
