/* This file is part of GTI (Generic Tool Infrastructure)
 *
 * Copyright (C)
 *  2008-2019 ZIH, Technische Universitaet Dresden, Federal Republic of Germany
 *  2008-2019 Lawrence Livermore National Laboratories, United States of America
 *  2013-2019 RWTH Aachen University, Federal Republic of Germany
 *
 * See the LICENSE file in the package base directory for details
 */

/**
 * @file GtiHelper.cpp
 *       @see gti::GtiHelper
 *
 * @author Felix M�nchhalfen
 *
 * @date 16.04.2014
 *
 */

#include "GtiHelper.h"

#include <mpi.h>
#include <pnmpimod.h>
#include <string>
#include <iostream>
#include <mutex>

#include <assert.h>
#include <stdio.h>
#include <stdint.h>

#include "pnmpi-config.h"

using namespace gti;

std::atomic<bool> GtiHelper::myInitedRank{false};
int GtiHelper::myRankInLayer = 0;
static std::recursive_mutex initMutex{};

//=============================
// GtiHelper
//=============================
GtiHelper::GtiHelper(void)
{ /*Nothing to do*/
}

//=============================
// ~GtiHelper
//=============================
GtiHelper::~GtiHelper(void)
{ /*Nothing to do*/
}

//=============================
// getFakedCommWorld
//=============================
inline MPI_Comm getFakedCommWorld()
{
    static MPI_Comm fakeCommWorld = MPI_COMM_NULL;
    static int inited, got_comm = 0;

    if (!got_comm) {
        // get rank of this process
        PMPI_Initialized(&inited);

        if (!inited)
            return fakeCommWorld; // TODO: what id to use in such a case ?

        int err;
        PNMPI_modHandle_t handle;
        PNMPI_Service_descriptor_t service;

        // We need to check whether MPI_COMM_WORLD was splited
        err = PNMPI_Service_GetModuleByName("split_processes", &handle);
        if (err == PNMPI_SUCCESS) {
            err = PNMPI_Service_GetServiceByName(handle, "SplitMod_getMySetComm", "p", &service);
            assert(err == PNMPI_SUCCESS);
            ((int (*)(void*))service.fct)(&fakeCommWorld);

        } else {
            // No splitting is active, use MPI_COMM_WORLD
            fakeCommWorld = MPI_COMM_WORLD;
        }
        got_comm = 1;
    }

    return fakeCommWorld;
}

//=============================
// buildLayer
//=============================
GtiTbonNodeInLayerId GtiHelper::buildLayer(bool threaded) { return buildLayerIdAsPureMpiLayer(); }

//=============================
// buildLayerIdAsPureMpiLayer
//=============================
GtiTbonNodeInLayerId GtiHelper::buildLayerIdAsPureMpiLayer()
{
    int rank = 0;

    if (!myInitedRank) {
        std::unique_lock<std::recursive_mutex> lock{initMutex};
        if (!myInitedRank) {
            MPI_Comm comm = getFakedCommWorld();

            if (comm != MPI_COMM_NULL)
                PMPI_Comm_rank(comm, &rank);

            myRankInLayer = rank;
            myInitedRank = true;
        }
    }

    return myRankInLayer;
}

//=============================
// getInstanceName
//=============================
GTI_RETURN GtiHelper::getInstanceName(const char** instanceName)
{
    // get self module, get name of own instance
    int err = PNMPI_SUCCESS;
    PNMPI_modHandle_t modHandle;

    err = PNMPI_Service_GetModuleSelf(&modHandle);
    assert(err == PNMPI_SUCCESS);

    /* get own module name */
    char temp[64];
    sprintf(temp, "instanceToUse");
    err = PNMPI_Service_GetArgument(modHandle, temp, instanceName);

    if (err != PNMPI_SUCCESS) {
        std::cerr << "Error: tool place module needs a PnMPI argument named \"instanceToUse\" that "
                     "lists a valid instance name to be used as instance."
                  << std::endl;
        return GTI_ERROR;
    }

    return GTI_SUCCESS;
}

/*EOF*/
