/* This file is part of GTI (Generic Tool Infrastructure)
 *
 * Copyright (C)
 *  2008-2019 ZIH, Technische Universitaet Dresden, Federal Republic of Germany
 *  2008-2019 Lawrence Livermore National Laboratories, United States of America
 *  2013-2019 RWTH Aachen University, Federal Republic of Germany
 *
 * See the LICENSE file in the package base directory for details
 */

/**
 * @file BuildGen.cpp
 *		@see gti::codegen::BuildGen
 *
 * @author Tobias Hilbrich
 * @date 19.08.2010
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iostream>
#include <list>
#include <assert.h>

#include "Verbose.h"

#include "BuildGen.h"
#include "config.h"

using namespace gti::codegen;

/**
 * Build file Generator (BuildGen) main function.
 */
int main(int argc, char** argv)
{
    //=========== 0 read input or print help ===========
    // print help if requested
    if ((argc == 2) && ((strncmp(argv[1], "--help", strlen("--help")) == 0) ||
                        (strncmp(argv[1], "-help", strlen("--help")) == 0) ||
                        (strncmp(argv[1], "-h", strlen("--help")) == 0))) {
        printUsage(argv[0], std::cout);
        return 0;
    }

    // enough arguments ?
    if (argc < 2) {
        std::cerr << "Error: Not enough arguments!" << std::endl << std::endl;
        printUsage(argv[0], std::cerr);
        return 1;
    }

    std::string fileName = argv[1];
    int retVal = 0;
    BuildGen generator(fileName, &retVal);

    return retVal;
}

//=============================
// printUsage
//=============================
void printUsage(std::string execName, std::ostream& out)
{
    out << "Usage: " << execName << " <BuildGeneratorInputXML>" << std::endl
        << std::endl
        << "E.g.: " << execName << " buildGenIn.xml" << std::endl;
}

//=============================
// BuildGen
//=============================
BuildGen::BuildGen(std::string inputFile, int* retVal)
{
    if (retVal)
        *retVal = 1;

    VERBOSE(1) << "Processing input file " << inputFile << " ..." << std::endl;

    // Open the input XML
    //=================================
    SpecificationNode currentPointer;
    SpecificationNode child, subchild;
    xmlDocPtr document;

    document = xmlParseFile(inputFile.c_str());

    if (document == NULL) {
        std::cerr << "Error loading input XML (" << inputFile << ")"
                  << "(" << __FILE__ << ":" << __LINE__ << ")" << std::endl;
        return;
    }

    currentPointer = xmlDocGetRootElement(document);

    if (currentPointer == NULL ||
        (xmlStrcmp(currentPointer()->name, (const xmlChar*)"buildgen-input") != 0)) {
        std::cerr << "Error: Document does not contains the root node (\"buildgen-input\")"
                  << "(" << __FILE__ << ":" << __LINE__ << ")" << std::endl;
        if (currentPointer)
            std::cerr << "Found \"" << currentPointer()->name << "\" instead!" << std::endl;
        xmlFreeDoc(document);
        return;
    }

    // Read: settings
    //=================================
    child = currentPointer.findChildNodeNamedOrErr(
        "settings",
        "|--> Error: root node has no \"settings\" child.");
    if (!child)
        return;
    if (!readSettings(child))
        return;

    //==Open the output build file
    /**
     * @todo the CMAKE_MODULE_PATH used here along with the
     * find_package commands is a bad solution, instead
     * we should include an installed cmake file that sets
     * the variables detected during the real cmake configure
     * for gti.
     */
    myBuildOut.open("CMakeLists.txt");
    myBuildOut << "##" << std::endl
               << "# Generated CMake build file for generated modules." << std::endl
               << "# Do not edit this file, it is generated, and likely " << std::endl
               << "# to be overwritten automatically." << std::endl
               << "##" << std::endl
               << std::endl
               << "CMAKE_MINIMUM_REQUIRED(VERSION 3.13.4...3.27.4)" << std::endl
               << "PROJECT (MUST_INTERMEDIATE C CXX)" << std::endl
               << "IF(NOT CMAKE_BUILD_TYPE)" << std::endl
               << "  SET(CMAKE_BUILD_TYPE \"RelWithDebInfo\" CACHE STRING \"Choose the type of "
                  "build, options are: None(CMAKE_CXX_FLAGS or CMAKE_C_FLAGS used) Debug Release "
                  "RelWithDebInfo MinSizeRel.\" FORCE)"
               << std::endl
               << "ENDIF(NOT CMAKE_BUILD_TYPE)" << std::endl
               << std::endl
               << "SET(CMAKE_MODULE_PATH \"" << myCMakeModPath << ";${CMAKE_MODULE_PATH}\")"
               << std::endl
               << "SET(LIBRARY_OUTPUT_PATH ${PROJECT_BINARY_DIR}/lib CACHE PATH \"Directory for "
                  "built libraries.\")"
               << std::endl
               << "SET(EXECUTABLE_OUTPUT_PATH ${PROJECT_BINARY_DIR}/bin CACHE PATH \"Directory for "
                  "built executables.\")"
               << std::endl
               << "MARK_AS_ADVANCED (LIBRARY_OUTPUT_PATH EXECUTABLE_OUTPUT_PATH)" << std::endl
               << "INCLUDE (HelperMacros)" << std::endl
               << "INCLUDE (InstallationConfiguration)" << std::endl
               << "INCLUDE (GTIConfig)" << std::endl
               << std::endl
               << "SET (CMAKE_CXX_STANDARD " << BUILDGEN_CXX_STANDARD << ")" << std::endl
               << "SET (CMAKE_CXX_STANDARD_REQUIRED TRUE)" << std::endl
               << "IF(CMAKE_INSTALL_PREFIX_INITIALIZED_TO_DEFAULT)" << std::endl
               << "    SET(CMAKE_INSTALL_PREFIX \"" << myModPath
               << "/..\" CACHE PATH \"Installation directory\" FORCE)" << std::endl
               << "ENDIF(CMAKE_INSTALL_PREFIX_INITIALIZED_TO_DEFAULT)" << std::endl
               << std::endl
               << "INCLUDE_DIRECTORIES(\"" << myModPath << "/../include\" \"" << myModPath
               << "/../src\" ${PnMPI_INCLUDE_DIR} ${MPI_C_INCLUDE_DIRS} ${GTI_TOOL_INCLUDE_PATH})"
               << std::endl
               << "IF(GTI_TOOL_LIB_PATH)" << std::endl
               << "    LINK_DIRECTORIES(${GTI_TOOL_LIB_PATH})" << std::endl
               << "ENDIF(GTI_TOOL_LIB_PATH)" << std::endl
               << std::endl;

    // Read: modules
    //=================================
    child = currentPointer.findChildNodeNamedOrErr(
        "modules",
        "|--> Error: root node has no \"modules\" child.");
    if (!child)
        return;

    subchild = child.findChildNodeNamed("module");
    while (subchild) {
        if (!readModule(subchild))
            return;

        // next
        subchild = subchild.findSiblingNamed("module");
    }

    // All successful
    myBuildOut.close();
    VERBOSE(1) << "--> SUCCESS" << std::endl;
    xmlFreeDoc(document);
    if (retVal)
        *retVal = 0;
}

//=============================
// ~BuildGen
//=============================
BuildGen::~BuildGen(void) {}

//=============================
// readSettings
//=============================
bool BuildGen::readSettings(SpecificationNode node)
{
    SpecificationNode child;

    //==Child: cmake-module-path
    child = node.findChildNodeNamedOrErr(
        "cmake-module-path",
        "|--> Error: a settings node has no \"cmake-module-path\" child.");
    if (!child)
        return false;
    myCMakeModPath = child.getNodeContent();

    //==Child: module-dir
    child = node.findChildNodeNamedOrErr(
        "module-dir",
        "|--> Error: a settings node has no \"module-dir\" child.");
    if (!child)
        return false;
    myModPath = child.getNodeContent();

    return true;
}

//=============================
// readSettings
//=============================
bool BuildGen::readModule(SpecificationNode node)
{
    SpecificationNode child, subchild, subsubchild;
    std::string source, moduleName, genOutName;
    std::list<std::string> extraSources, extraLibs;

    std::list<std::string>::iterator iter;

    //==Child: main-source
    child = node.findChildNodeNamedOrErr(
        "main-source",
        "|--> Error: a module node has no \"main-source\" child.");
    if (!child)
        return false;
    source = child.getNodeContent();

    //==Child: module-name
    child = node.findChildNodeNamedOrErr(
        "module-name",
        "|--> Error: a module node has no \"module-name\" child.");
    if (!child)
        return false;
    moduleName = child.getNodeContent();

    //==Child: gen-output
    child = node.findChildNodeNamedOrErr(
        "gen-output",
        "|--> Error: a module node has no \"gen-output\" child.");
    if (!child)
        return false;
    genOutName = child.getNodeContent();

    //==Open the gen-output XML
    xmlDocPtr document;

    document = xmlParseFile(genOutName.c_str());

    if (document == NULL) {
        std::cerr << "Error loading input XML of a module (" << genOutName << ")"
                  << "(" << __FILE__ << ":" << __LINE__ << ")" << std::endl;
        return false;
    }

    child = xmlDocGetRootElement(document);

    if (child == NULL || ((xmlStrcmp(child()->name, (const xmlChar*)"recvgen-output") != 0) &&
                          (xmlStrcmp(child()->name, (const xmlChar*)"wrappgen-output") != 0))) {
        std::cerr << "Error: Document does not contains the root node \"recvgen-output\" or "
                     "\"wrappgen-output\" "
                  << "(" << __FILE__ << ":" << __LINE__ << ")" << std::endl;
        if (child)
            std::cerr << "Found \"" << child()->name << "\" instead!" << std::endl;
        xmlFreeDoc(document);
        return false;
    }

    //==Read extra libs
    subchild = child.findChildNodeNamedOrErr(
        "extra-sources",
        "|--> Error: a module gen output has no \"extra-sources\" child.");
    if (!subchild)
        return false;

    subsubchild = subchild.findChildNodeNamed("source");
    while (subsubchild) {
        extraSources.push_back(subsubchild.getNodeContent());

        //==next
        subsubchild = subsubchild.findSiblingNamed("source");
    }

    //==Read extra sources
    subchild = child.findChildNodeNamedOrErr(
        "extra-libs",
        "|--> Error: a module gen output has no \"extra-libs\" child.");
    if (!subchild)
        return false;

    subsubchild = subchild.findChildNodeNamed("lib");
    while (subsubchild) {
        extraLibs.push_back(subsubchild.getNodeContent());

        //==next
        subsubchild = subsubchild.findSiblingNamed("lib");
    }

    //==Add the module to the CMake file
    myBuildOut << "###-------------------- " << moduleName << " ---------------------###"
               << std::endl
               << "SET (BASE_SOURCES " << myModPath << "/../src/GtiHelper.cpp)" << std::endl
               << std::endl
               << "GTI_MAC_ADD_MODULE_NO_COVERAGE (" << moduleName << " \"" << source << ";";

    for (iter = extraSources.begin(); iter != extraSources.end(); iter++) {
        myBuildOut << *iter << ";";
    }

    myBuildOut << "${BASE_SOURCES}\" \"CXX\")" << std::endl
               << std::endl
               << "TARGET_LINK_LIBRARIES (" << moduleName;

    for (iter = extraLibs.begin(); iter != extraLibs.end(); iter++) {
        myBuildOut << " " << *iter << " ";
    }

    myBuildOut << ")" << std::endl << std::endl;

    //==Clean up
    xmlFreeDoc(document);

    return true;
}

/*EOF*/
