/* This file is part of GTI (Generic Tool Infrastructure)
 *
 * Copyright (C)
 *  2008-2019 ZIH, Technische Universitaet Dresden, Federal Republic of Germany
 *  2008-2019 Lawrence Livermore National Laboratories, United States of America
 *  2013-2023 RWTH Aachen University, Federal Republic of Germany
 *
 * See the LICENSE file in the package base directory for details
 */

#include <mpi.h>

#include "gtiConfig.h"
#include <pnmpi/service.h>
#include <pnmpimod.h>

#if defined(GTI_ENABLE_SESSIONS)
#include "SplitProcessesModuleProxy.h"
#endif

namespace
{

using Callback = void();

auto get_registration_func() -> void (*)(Callback*)
{
    auto startupModule = PNMPI_modHandle_t{PNMPI_MODHANDLE_NULL};
    int err = PNMPI_Service_GetModuleByName("libstartupReceiver", &startupModule);
    if (err != PNMPI_SUCCESS) {
        return nullptr;
    }

    PNMPI_Service_descriptor_t service;
    err = PNMPI_Service_GetServiceByName(startupModule, "OnInit", "p", &service);
    if (err != PNMPI_SUCCESS) {
        return nullptr;
    }

    return reinterpret_cast<void (*)(Callback*)>(service.fct);
}

const auto on_init = get_registration_func();

void do_init()
{
    int initialized = 0;
    MPI_Initialized(&initialized);

    // It is allowed to call init with different thread levels across the processes. We do
    // not need to match with the init calls of the application.
    // See: MPI 4.0 Standard, p.492, l.3
    if (!static_cast<bool>(initialized)) {
        MPI_Init(nullptr, nullptr);
    }
}

} // namespace

int main()
{
#if defined(GTI_ENABLE_SESSIONS)
    // With Sessions enabled we call MPI_Init lazily when the application did.
    if (on_init != nullptr) {
        on_init(do_init);
    }

    auto splitMod = gti::SplitProcessesModuleProxy{};
    splitMod.runTool();
#else
    do_init();
#endif

    MPI_Abort(MPI_COMM_WORLD, 1);
    return 0;
}
