/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_RMAValueChecks.h
 *       @see I_RMAValueChecks.
 *
 *  @date 15.01.2025
 *  @author Cornelius Pätzold
 */

#include "I_Module.h"
#include "GtiEnums.h"
#include "BaseIds.h"
#include "MustTypes.h"

#ifndef I_RMAVALUECHECKS_H
#define I_RMAVALUECHECKS_H

/**
 * Interface for correctness checks for RMA values.
 *
 * Dependencies (order as listed):
 * - ParallelIdAnalysis
 * - CreateMessage
 * - ArgumentAnalysis
 * - BaseConstants
 *
 */
class I_RMAValueChecks : public gti::I_Module
{
  public:
    /**
     *	Initializes the aggregated values of valid assertions.
     */
    virtual gti::GTI_ANALYSIS_RETURN init() = 0;

    /**
     *	Checks if a MPI_Win_fence assertion is valid.
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param assert to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfInvalidFenceAssertion(MustParallelId pId, MustLocationId lId, int assert) = 0;

    /**
     *	Checks if a MPI_Win_post assertion is valid.
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param assert to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfInvalidPostAssertion(MustParallelId pId, MustLocationId lId, int assert) = 0;

    /**
     *	Checks if a MPI_Win_start assertion is valid.
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param assert to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfInvalidStartAssertion(MustParallelId pId, MustLocationId lId, int assert) = 0;

    /**
     *	Checks if a MPI_Win_lock or MPI_Win_lock_all assertion is valid.
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param assert to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfInvalidLockAssertion(MustParallelId pId, MustLocationId lId, int assert) = 0;

    /**
     *	Checks if the lock type provided MPI_Win_lock is MPI_LOCK_EXCLUSIVE or MPI_LOCK_SHARED.
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param assert to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfInvalidLockType(MustParallelId pId, MustLocationId lId, int lock_type) = 0;
};

#endif /*I_RMAValueChecks*/
