/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef MUST_DEBUG_H_
#define MUST_DEBUG_H_

#include <ostream>

typedef enum {
    MUST_LOG_NONE = 0,
    MUST_LOG_FATAL, /* Errors that should be reported to must-feedback */
    MUST_LOG_ERROR, /* Errors that are important to the user but that are not fatal */
    MUST_LOG_DEV_ERROR,
    MUST_LOG_WARN,
    MUST_LOG_INFO,
    MUST_LOG_TRACE
} mustDebugLogLevel;

typedef enum {
    MUST_INIT = 1 << 1,
    MUST_DLP2P = 1 << 2,
    MUST_DLCOLL = 1 << 3,
    MUST_CHECK = 1 << 4,
    MUST_TRACK = 1 << 5,
    MUST_TSAN = 1 << 6,
    MUST_FIN = 1 << 7,
    MUST_MSG = 1 << 8,
    MUST_UNMAPPED = 1 << 9,
    MUST_TYPEART = 1 << 10,
    MUST_DLOTHER = 1 << 11,
    MUST_OMP = 1 << 12,
    MUST_ALL = ~0
} mustDebugLogSubSys;

std::ostream& mustDebugLog(
    mustDebugLogLevel level,
    unsigned long mask,
    const char* function,
    const char* file,
    int line);

bool mustDebugDisplayMsg(mustDebugLogLevel level, unsigned long mask);

void mustDebugAbortOnFatal(
    mustDebugLogLevel level,
    unsigned long mask,
    const char* function,
    const char* file,
    int line);

#define LOG(level, mask, msg)                                                                      \
    if (mustDebugDisplayMsg(level, mask))                                                          \
        mustDebugLog(level, mask, __FUNCTION__, __FILE__, __LINE__) << msg << "\n";                \
    mustDebugAbortOnFatal(level, mask, __FUNCTION__, __FILE__, __LINE__);
#define FATAL(mask, msg) LOG(MUST_LOG_FATAL, mask, msg)
#define ERROR(mask, msg) LOG(MUST_LOG_ERROR, mask, msg)
#define DEV_ERROR(mask, msg) LOG(MUST_LOG_DEV_ERROR, mask, msg)
#define WARN(mask, msg) LOG(MUST_LOG_WARN, mask, msg)
#define INFO(mask, msg) LOG(MUST_LOG_INFO, mask, msg)
#define TRACE(mask, msg) LOG(MUST_LOG_TRACE, mask, msg)

#endif
