/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file QOpCommunicationGroup.cpp
 *       @see must::QOpCommunicationGroup.
 *
 *  @date 22.04.2025
 *  @author Cornlius Pätzold
 */

#include "QOpCommunicationGroup.h"

#include "DWaitState.h"

using namespace must;

//=============================
// QOpCommunicationGroup
//=============================
QOpCommunicationGroup::QOpCommunicationGroup(
    DWaitState* dws,
    MustParallelId pId,
    MustLocationId lId,
    MustLTimeStamp ts,
    I_CommPersistent* comm,
    I_WinPersistent* win,
    bool isSend,
    int sourceTarget,
    MustSendMode mode)
    : QOp(dws, pId, lId, ts),
      QOpCommunicationP2P(dws, pId, lId, ts, comm, isSend, sourceTarget, false, mode, 0), myWin(win)
{
    // Nothing to do
}

//=============================
// ~QOpCommunicationGroup
//=============================
QOpCommunicationGroup::~QOpCommunicationGroup(void)
{
    if (myWin) {
        myWin->erase();
    }
    myWin = NULL;
}

std::string QOpCommunicationGroup::getCompletionEdgeLabel()
{
    // MPI_Win_start and MPI_Win_wait
    if (myWin)
        return "group";

    // Neighborhood collectives
    return "neighbor";
}

std::stringstream QOpCommunicationGroup::getLabels(
    int count,
    std::map<I_Comm*, std::string>& commLabels,
    std::map<I_Win*, std::string>& winLabels)
{
    //==Prepare comm label
    std::string commLabel = "";
    I_Comm* cInfo;
    std::string winLabel = "";

    std::map<I_Comm*, std::string>::iterator commIter;
    for (commIter = commLabels.begin(); commIter != commLabels.end(); commIter++) {
        if (myComm->compareComms(commIter->first)) {
            cInfo = commIter->first;
            commLabel = commIter->second;
            break;
        }
    }

    if (myWin) {
        std::map<I_Win*, std::string>::iterator winIter;
        for (winIter = winLabels.begin(); winIter != winLabels.end(); winIter++) {
            if (myWin->compareWins(winIter->first)) {
                winLabel = winIter->second;
                break;
            }
        }
    }

    std::stringstream labels;
    std::string commTopology = "";
    if (cInfo->isGraph())
        commTopology = "graph-";
    else if (cInfo->isCartesian())
        commTopology = "cart-";

    for (int i = 0; i < count; i++) {
        if (myWin)
            labels << "win=" << winLabel << ", ";
        labels << commTopology << "comm=" << commLabel;
    }
    return labels;
}

//=============================
// getUsedWins
//=============================
std::list<I_Win*> QOpCommunicationGroup::getUsedWins(void)
{
    checkAlive();
    std::list<I_Win*> ret;
    if (myWin)
        ret.push_back(myWin);
    return ret;
}

/*EOF*/
