/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file DGroupOp.h
 *       @see must::DGroupOp.
 *
 *  @date 20.01.2012
 *  @author Tobias Hilbrich, Joachim Protze, Mathias Korepkat, Fabian Haensel, Cornelius Pätzold
 */

#include "MustTypes.h"
#include "I_DOperation.h"
#include "I_CommTrack.h"
#include "I_RequestTrack.h"
#include "I_DatatypeTrack.h"
#include "I_CommTrack.h"
#include "I_WinTrack.h"
#include "I_DatatypeTrack.h"
#include <vector>

#ifndef DGROUPOP_H
#define DGROUPOP_H

#include "DGroupMatch.h"

using namespace gti;

namespace must
{

class DGroupOpPartnerInfo
{
  public:
    DGroupOp* myOp;
    int myTargetRank;
    MustDatatypeType myTypeHandle;
    I_DatatypePersistent* myType;
    int myCount;
    MustLTimeStamp myTimestamp;
    bool myIsSend;

    DGroupOpPartnerInfo(DGroupOp* op, int targetRank, MustLTimeStamp timestamp, bool isSend);

    DGroupOpPartnerInfo(
        DGroupOp* op,
        int targetRank,
        MustDatatypeType typeHandle,
        I_DatatypePersistent* dInfo,
        int count,
        MustLTimeStamp timestamp,
        bool isSend);

    ~DGroupOpPartnerInfo();

    bool matchTypes(DGroupOpPartnerInfo* other);
};

/**
 * A group operation.
 */
class DGroupOp : public I_DOperation
{

  public:
    DGroupMatch* myMatcher;

  protected:
    MustParallelId myPId; /**< Information on context that created the send/recv.*/
    MustLocationId myLId; /**< Information on context that created the send/recv.*/
    int myRank;           /**< MPI_COMM_WORLD rank that issued this op.*/
    std::vector<DGroupOpPartnerInfo*> myPartnerInfos;
    I_Destructable* myPersistentHandle; /**< The win or comm of the communication.*/
    int myCallId;                       /**< Id to distinguish between MPI calls*/
    bool myIsSend; /**< True if this is a send, false otherwise, only used where this information is
                      required.*/
    bool myIsCollective; /**< True if this is a collective operation, false otherwise. */

    /**
     * Private constructor that copies the given op.
     */
    DGroupOp(DGroupOp* from);

  public:
    /**
     * Constructor for no transfer send/recv communication with a vector of world ranks.
     * Control of the datatype and win is given to this class
     * and will be erased in its destuctor.
     */
    DGroupOp(
        MustParallelId pId,
        MustLocationId lId,
        I_Destructable* persistentHandle,
        DGroupMatch* matcher,
        bool isSend,
        std::vector<int> ranks,
        std::vector<MustLTimeStamp> timestamps,
        int callId,
        bool isCollective);

    // Same type and sendCount
    DGroupOp(
        MustParallelId pId,
        MustLocationId lId,
        I_Destructable* persistentHandle,
        DGroupMatch* matcher,
        bool isSend,
        std::vector<int> ranks,
        std::vector<MustLTimeStamp> timestamps,
        MustDatatypeType typeHandle,
        I_DatatypePersistent* dInfo,
        int sendCount,
        int callId,
        bool isCollective);

    // Same type but multiple sendCounts
    DGroupOp(
        MustParallelId pId,
        MustLocationId lId,
        I_Destructable* persistentHandle,
        DGroupMatch* matcher,
        bool isSend,
        std::vector<int> ranks,
        std::vector<MustLTimeStamp> timestamps,
        MustDatatypeType typeHandle,
        I_DatatypePersistent* dInfo,
        std::vector<int> sendCounts,
        int callId,
        bool isCollective);

    // Multiple types and sendCounts
    DGroupOp(
        MustParallelId pId,
        MustLocationId lId,
        I_Destructable* persistentHandle,
        DGroupMatch* matcher,
        bool isSend,
        std::vector<int> ranks,
        std::vector<MustLTimeStamp> timestamps,
        std::vector<MustDatatypeType> typeHandles,
        std::vector<I_DatatypePersistent*> dInfos,
        std::vector<int> sendCounts,
        int callId,
        bool isCollective);

    /**
     * Destructor.
     * Erases the datatype and comm.
     */
    ~DGroupOp(void);

    /**
     * @see I_Operation::process
     */
    PROCESSING_RETURN process(int rank);

    /**
     * @see I_Operation::print
     */
    GTI_RETURN print(std::ostream& out);

    /**
     * Returns a persistent hanlde copied from
     * this ops handle.
     * @return copy of myPersistentHandle.
     */
    I_Destructable* getPersistentHandleCopy(void);

    /**
     * Returns the persistent hanlde.
     * @return myPersistentHandle.
     */
    I_Destructable* getPersistentHandle(void);

    /**
     * Compares the handle of this op with an other op.
     * @return true iff the this ops handle and others hanlde are the same, false otherwise.
     */
    bool compareHandles(DGroupOp* other);

    /**
     * Compares the handle of this op with another handle.
     * @return true iff the this ops handle and others hanlde are the same, false otherwise.
     */
    bool compareHandles(I_Destructable* other);

    /**
     * Returns the vector of world ranks this op sends to or recv from.
     */
    std::vector<DGroupOpPartnerInfo*>& getPartnerInfos(void);

    /**
     * Returns true iff this is a send.
     * @return true iff send.
     */
    bool isSend(void);

    /**
     * Returns true iff this is a collective operation.
     * @return true iff collective operation.
     */
    bool isCollective(void);

    /**
     * Returns the call id of this op.
     * @return call id.
     */
    int getCallId(void);

    /**
     * Returns the rank that issued this operation.
     */
    int getIssuerRank(void);

    /**
     * Returns parallel id of this op.
     * @return parallel id.
     */
    MustParallelId getPId(void);

    /**
     * Returns location id of thi op.
     * @return location id.
     */
    MustLocationId getLId(void);

    /**
     * creates a copy of this op.
     */
    DGroupOp* copy(void);

    void printHandleInfo(
        std::stringstream& stream,
        std::list<std::pair<MustParallelId, MustLocationId>>* references);

    bool printCollectiveMismatch(DGroupOp* other);
};

} /*namespace must*/

#endif /*DGROUPOP_H*/
