/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_DGroupMatch.h
 *       @see I_DGroupMatch.
 *
 *  @date 20.01.2012
 *  @author Tobias Hilbrich, Mathias Korepkat, Joachim Protze, Fabian Haensel
 */

#include "I_Module.h"
#include "GtiEnums.h"
#include "BaseIds.h"
#include "I_ChannelId.h"
#include "MustEnums.h"
#include "I_Comm.h"
#include "I_Win.h"
#include "I_DGroupListener.h"

#ifndef I_DGROUPMATCH_H
#define I_DGROUPMATCH_H

/**
 * Interface for a group comunication message matching.
 *
 * Dependencies (order as listed):
 * - ParallelIdAnalysis
 * - LocationAnalysis
 * - BaseConstants
 * - CreateMessage  ##for printing lost messages
 * - CommTrack
 * - RequestTrack
 * - DatatypeTrack
 * - FloodControl
 * - Profiler
 *
 */
class I_DGroupMatch : public gti::I_Module
{
  public:
    /**
     * Notification of MPI_Init.
     *
     * @param pId parallel Id of the call site.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN init(MustParallelId pId) = 0;

    /**
     * Notification of a started send.
     *
     * @param pId parallel Id of the call site.
     * @param lId location Id of the call site.
     * @param group to send to.
     * @param win for the send.
     * @param callId id of the MPI call, @see must::MustCollCommType.
     * @param ts if this is an op from remote, this is
     *              the associated timestamp from the remote
     *              side, otherwise it is some value.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN post(
        MustParallelId pId,
        MustLocationId lId,
        MustGroupType group,
        MustWinType win,
        int callId) = 0;

    /**
     * Notification of a started recv.
     *
     * @param pId parallel Id of the call site.
     * @param lId location Id of the call site.
     * @param group to receive from.
     * @param win for the receive.
     * @param callId id of the MPI call, @see must::MustCollCommType.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN start(
        MustParallelId pId,
        MustLocationId lId,
        MustGroupType group,
        MustWinType win,
        int callId) = 0;

    /**
     *
     *
     * @param pId parallel Id of the call site.
     * @param lId location Id of the call site.
     * @param win for the receive.
     * @param callId id of the MPI call, @see must::MustCollCommType.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    complete(MustParallelId pId, MustLocationId lId, MustWinType win, int callId) = 0;

    /**
     *
     *
     * @param pId parallel Id of the call site.
     * @param lId location Id of the call site.
     * @param win for the receive.
     * @param callId id of the MPI call, @see must::MustCollCommType.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    wait(MustParallelId pId, MustLocationId lId, MustWinType win, int callId) = 0;

    virtual gti::GTI_ANALYSIS_RETURN recvPassToGroupNoTransfer(
        MustParallelId pId,
        MustLocationId lId,
        const int* worldRanks,
        const MustLTimeStamp* timestamps,
        int worldRanksCount,
        int handleType,
        MustWinType win,
        int callId,
        bool isCollective) = 0;

    virtual gti::GTI_ANALYSIS_RETURN recvPassToGroupN(
        MustParallelId pId,
        MustLocationId lId,
        MustDatatypeType type,
        int sendCount,
        const int* worldRanks,
        const MustLTimeStamp* timestamps,
        int count,
        int handleType,
        MustRemoteIdType remoteHandleId,
        int callId,
        bool isCollective) = 0;

    virtual gti::GTI_ANALYSIS_RETURN recvPassToGroupTypes(
        MustParallelId pId,
        MustLocationId lId,
        MustDatatypeType* types,
        const int* sendCounts,
        const int* worldRanks,
        const MustLTimeStamp* timestamps,
        int count,
        int handleType,
        MustRemoteIdType remoteHandleId,
        int callId,
        bool isCollective) = 0;

    virtual gti::GTI_ANALYSIS_RETURN recvPassToGroupCounts(
        MustParallelId pId,
        MustLocationId lId,
        MustDatatypeType type,
        const int* sendCounts,
        const int* worldRanks,
        const MustLTimeStamp* timestamps,
        int count,
        int handleType,
        MustRemoteIdType remoteHandleId,
        int callId,
        bool isCollective) = 0;

    /**
     * Registers a point to point operation listener for this module.
     */
    virtual void registerListener(must::I_DGroupListener* listener) = 0;

    virtual gti::GTI_ANALYSIS_RETURN nbrSendN(
        MustParallelId pId,
        MustLocationId lId,
        MustCommType comm,
        int callId,
        int sendcount,
        MustDatatypeType sendtype) = 0;

    virtual gti::GTI_ANALYSIS_RETURN nbrSendCounts(
        MustParallelId pId,
        MustLocationId lId,
        MustCommType comm,
        int callId,
        const int* sendcounts,
        MustDatatypeType sendtype) = 0;

    virtual gti::GTI_ANALYSIS_RETURN nbrSendTypes(
        MustParallelId pId,
        MustLocationId lId,
        MustCommType comm,
        int callId,
        const int* sendcounts,
        const MustDatatypeType* sendtypes) = 0;

    virtual gti::GTI_ANALYSIS_RETURN nbrRecvN(
        MustParallelId pId,
        MustLocationId lId,
        MustCommType comm,
        int callId,
        int recvcount,
        MustDatatypeType recvtype,
        bool hasRequest,
        MustRequestType request) = 0;

    virtual gti::GTI_ANALYSIS_RETURN nbrRecvCounts(
        MustParallelId pId,
        MustLocationId lId,
        MustCommType comm,
        int callId,
        const int* recvcounts,
        MustDatatypeType recvtype,
        bool hasRequest,
        MustRequestType request) = 0;

    virtual gti::GTI_ANALYSIS_RETURN nbrRecvTypes(
        MustParallelId pId,
        MustLocationId lId,
        MustCommType comm,
        int callId,
        const int* recvcounts,
        const MustDatatypeType* recvtypes,
        bool hasRequest,
        MustRequestType request) = 0;

}; /*class I_DGroupMatch*/

#endif /*I_DGroupMATCH_H*/
