/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file FileChecks.cpp
 *       @see MUST::FileChecks.
 */

#include "GtiMacros.h"
#include "MustEnums.h"
#include "PrefixedOstream.hpp"

#include "FileChecks.h"

#include <sstream>

using namespace must;

mGET_INSTANCE_FUNCTION(FileChecks)
mFREE_INSTANCE_FUNCTION(FileChecks)
mPNMPI_REGISTRATIONPOINT_FUNCTION(FileChecks)

//=============================
// Constructor
//=============================
FileChecks::FileChecks(const char* instanceName)
    : gti::ModuleBase<FileChecks, I_FileChecks>(instanceName)
{
    // create sub modules
    std::vector<I_Module*> subModInstances;
    subModInstances = createSubModuleInstances();

    // handle sub modules
#define NUM_SUBMODULES 4
    if (subModInstances.size() < NUM_SUBMODULES) {
        must::cerr << "Module has not enough sub modules, check its analysis specification! ("
                   << __FILE__ << "@" << __LINE__ << ")" << std::endl;
        assert(0);
    }
    if (subModInstances.size() > NUM_SUBMODULES) {
        for (std::vector<I_Module*>::size_type i = NUM_SUBMODULES; i < subModInstances.size(); i++)
            destroySubModuleInstance(subModInstances[i]);
    }

    myPIdMod = (I_ParallelIdAnalysis*)subModInstances[0];
    myLogger = (I_CreateMessage*)subModInstances[1];
    myArgMod = (I_ArgumentAnalysis*)subModInstances[2];
    myFileMod = (I_FileTrack*)subModInstances[3];

    // Initialize module data
    // Nothing to do
}

//=============================
// Destructor
//=============================
FileChecks::~FileChecks()
{
    if (myPIdMod)
        destroySubModuleInstance((I_Module*)myPIdMod);
    myPIdMod = NULL;

    if (myLogger)
        destroySubModuleInstance((I_Module*)myLogger);
    myLogger = NULL;

    if (myArgMod)
        destroySubModuleInstance((I_Module*)myArgMod);
    myArgMod = NULL;

    if (myFileMod)
        destroySubModuleInstance((I_Module*)myFileMod);
    myFileMod = NULL;
}

GTI_ANALYSIS_RETURN
FileChecks::errorIfNull(MustParallelId pId, MustLocationId lId, int aId, MustFileType file)
{
    auto info = myFileMod->getFile(pId, file);

    if (info && info->isNull()) {
        std::stringstream stream;
        stream << "Argument " << myArgMod->getIndex(aId) << " (" << myArgMod->getArgName(aId)
               << ") is MPI_FILE_NULL where a valid communicator was expected.";

        myLogger->createMessage(MUST_ERROR_FILE_NULL, pId, lId, MustErrorMessage, stream.str());
        return GTI_ANALYSIS_FAILURE;
    }

    return GTI_ANALYSIS_SUCCESS;
}

GTI_ANALYSIS_RETURN
FileChecks::errorIfNotKnown(MustParallelId pId, MustLocationId lId, int aId, MustFileType file)
{
    auto info = myFileMod->getFile(pId, file);

    if (info == NULL) {
        std::stringstream stream;
        stream << "Argument " << myArgMod->getIndex(aId) << " (" << myArgMod->getArgName(aId)
               << ") is an unknown file where a valid file was expected.";

        myLogger->createMessage(MUST_ERROR_FILE_UNKNWOWN, pId, lId, MustErrorMessage, stream.str());
        return GTI_ANALYSIS_FAILURE;
    }

    return GTI_ANALYSIS_SUCCESS;
}

GTI_ANALYSIS_RETURN
FileChecks::errorIfOpenRequest(MustParallelId pId, MustLocationId lId, MustFileType file)
{
    auto info = myFileMod->getFile(pId, file);
    if (!info)
        return GTI_ANALYSIS_SUCCESS;

    auto& reqs = info->getOpenRequests();
    if (reqs.size()) {
        std::stringstream stream;
        std::list<std::pair<MustParallelId, MustLocationId>> refs;
        info->printInfo(stream, &refs);
        stream << " has uncompleted IO operations, displayed at the remaining "
                  "references.";
        for (const auto& req : reqs)
            refs.emplace_back(pId, req.second);

        myLogger->createMessage(
            MUST_ERROR_FILE_UNCOMPLETED_REQUESTS,
            pId,
            lId,
            MustErrorMessage,
            stream.str(),
            refs);
        return GTI_ANALYSIS_FAILURE;
    }

    return GTI_ANALYSIS_SUCCESS;
}

/*EOF*/
