/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_InfoChecks.h
 *       @see I_InfoChecks.
 *
 *  @date 14.04.2011
 *  @author Mathias Korepkat
 */

#include "I_Module.h"
#include "GtiEnums.h"
#include "BaseIds.h"
#include "I_InfoTrack.h"

#ifndef I_INFOCHECKS_H
#define I_INFOCHECKS_H

/**
 * Interface for correctness checks of info handles.
 *
 * Dependencies (order as listed):
 * - ParallelIdAnalysis
 * - CreateMessage
 * - ArgumentAnalysis
 * - InfoTrack
 * - BaseConstants
 */
class I_InfoChecks : public gti::I_Module
{
  public:
    /**
     *	Checks if the info hanldle is MPI_INFO_NULL, if so,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param infoHandle to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfNull(MustParallelId pId, MustLocationId lId, int aId, MustInfoType infoHandle) = 0;

    /**
     *	Checks if the info hanldle is known, if not,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param infoHandle to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfNotKnown(MustParallelId pId, MustLocationId lId, int aId, MustInfoType infoHandle) = 0;

    /**
     *	Checks if the length of a key is within a range of 0 to MPI_MAX_INFO_KEY, if not,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param key to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN errorIfLengthNotWithinRangeZeroAndLessMaxInfoKey(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        char* key) = 0;

    /**
     *	Checks if the length of a value is within a range of 0 to MPI_MAX_INFO_VAL, if not,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param value to check.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN errorIfLengthNotWithinRangeZeroAndLessMaxInfoVal(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        char* value) = 0;

    /**
     *	Checks if a key is defined in the info handle, if not,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param key to check.
     *	@param infoHandle the info hanlde to check
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN errorIfKeyNotDefined(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        char* key,
        MustInfoType infoHandle) = 0;

    /**
     *	Checks if a key is defined in the info handle, if not,
     *	manifests as warning.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param key to check.
     *	@param infoHandle the info hanlde to check
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN warningIfKeyNotDefined(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        char* key,
        MustInfoType infoHandle) = 0;

    /**
     *	Checks if the nth key is defined in the info handle, if not,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param n order of key to check.
     *	@param infoHandle the info hanlde to check
     *	@return see gti::GTI_ANALYSIS_RETURN.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN errorIfNthKeyNotDefined(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        int n,
        MustInfoType infoHandle) = 0;

    /**
     *	Checks if the nth key is defined in the info handle, if not,
     *	manifests as warning.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the info to check.
     *	@param n order of key to check.
     *	@param infoHandle the info hanlde to check
     *	@return see gti::GTI_ANALYSIS_RETURN.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN warningIfNthKeyNotDefined(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        int n,
        MustInfoType infoHandle) = 0;

}; /*class I_InfoChecks*/

#endif /*I_INFOCHECKS_H*/
