/**
 * @file MpiTSanAnnotations.cpp
 * 	@see MpiTSanAnnotations.
 */

#include "GtiEnums.h"
#include "GtiMacros.h"
#include "MpiTSanAnnotations.h"
#include "MustEnums.h"
#include "MustDefines.h"

#include <sstream>
#include <fstream>

using namespace must;

mGET_INSTANCE_FUNCTION(MpiTSanAnnotations)
mFREE_INSTANCE_FUNCTION(MpiTSanAnnotations)
mPNMPI_REGISTRATIONPOINT_FUNCTION(MpiTSanAnnotations)

//=============================
// Constructor
//=============================
MpiTSanAnnotations::MpiTSanAnnotations(const char* instanceName)
    : gti::ModuleBase<MpiTSanAnnotations, I_MpiTSanAnnotations, true>(instanceName),
      requestBlocklist(), persistentBlocklist(), requestLocation()
{
    // create sub modules
    std::vector<I_Module*> subModInstances;
    subModInstances = createSubModuleInstances();

    // handle sub modules
#define NUM_SUBMODULES 7
    if (subModInstances.size() < NUM_SUBMODULES) {
        std::cerr << "Module does not have enough sub modules, check its analysis specification! ("
                  << __FILE__ << "@" << __LINE__ << ")" << std::endl;
        assert(0);
    }
    if (subModInstances.size() > NUM_SUBMODULES) {
        for (std::vector<I_Module*>::size_type i = NUM_SUBMODULES; i < subModInstances.size();
             ++i) {
            destroySubModuleInstance(subModInstances[i]);
        }
    }

    myPIdMod = (I_ParallelIdAnalysis*)subModInstances[0];
    myLIdMod = (I_LocationAnalysis*)subModInstances[1];
    myLogger = (I_CreateMessage*)subModInstances[2];
    myArgMod = (I_ArgumentAnalysis*)subModInstances[3];
    myDatMod = (I_DatatypeTrack*)subModInstances[4];
    myReqMod = (I_RequestTrack*)subModInstances[5];
    myTSanMod = (I_TSan*)subModInstances[6];
}

//=============================
// Destructor
//=============================
MpiTSanAnnotations::~MpiTSanAnnotations(void)
{
    if (myPIdMod)
        destroySubModuleInstance((I_Module*)myPIdMod);
    myPIdMod = NULL;

    if (myLIdMod)
        destroySubModuleInstance((I_Module*)myLIdMod);
    myLIdMod = NULL;

    if (myLogger)
        destroySubModuleInstance((I_Module*)myLogger);
    myLogger = NULL;

    if (myArgMod)
        destroySubModuleInstance((I_Module*)myArgMod);
    myArgMod = NULL;

    if (myDatMod)
        destroySubModuleInstance((I_Module*)myDatMod);
    myDatMod = NULL;

    if (myReqMod)
        destroySubModuleInstance((I_Module*)myReqMod);
    myReqMod = NULL;

    if (myTSanMod)
        destroySubModuleInstance((I_Module*)myTSanMod);
    myTSanMod = NULL;
}

// The const kPCInc must be in sync with StackTrace::GetPreviousInstructionPc
#if defined(__powerpc64__) || defined(__arm__) || defined(__aarch64__)
// PCs are always 4 byte aligned.
const int kPCInc = 4;
#elif defined(__sparc__) || defined(__mips__)
const int kPCInc = 8;
#else
const int kPCInc = 1;
#endif

//=============================
// annotateFromMemIntervalList.
//=============================
#define TSAN_MARK(a) ((a + 1) | 1ULL << 60)
// #define TSAN_MARK(a) (a)
void MpiTSanAnnotations::annotateFromMemIntervalList(
    MustParallelId pId,
    MustLocationId lId,
    const MustMemIntervalListType& iList,
    void* pc = nullptr,
    void* fp = nullptr)
{
    assert(iList.size());
    if (!pc) {
        PNMPI_Service_GetReturnAddress(&pc);
    }
    if (!fp) {
        PNMPI_Service_GetFunctionAddress(&fp);
    }

    // Function entry
    myTSanMod->annotateFuncEntry(pc);
    for (auto stridedBlock : iList) {
#if defined(MUST_DEBUG)
        std::cout << "Annotation: isSend=" << stridedBlock.isSend << ", first=" << std::hex
                  << stridedBlock.first << ", second=" << std::hex << stridedBlock.second
                  << std::endl;
#endif
        if (stridedBlock.isSend) {
            if (stridedBlock.stride == 0) {
                myTSanMod->annotateMemoryReadPC(
                    pId,
                    lId,
                    stridedBlock.first,
                    stridedBlock.blocksize,
                    ((char*)fp) + kPCInc);
            } else {
                for (MustAddressType addr = stridedBlock.first; addr < stridedBlock.second;
                     addr += stridedBlock.stride) {
                    myTSanMod->annotateMemoryReadPC(
                        pId,
                        lId,
                        addr,
                        stridedBlock.blocksize,
                        ((char*)fp) + kPCInc);
                }
            }
        } else {
            if (stridedBlock.stride == 0) {
                myTSanMod->annotateMemoryWritePC(
                    pId,
                    lId,
                    stridedBlock.first,
                    stridedBlock.blocksize,
                    ((char*)fp) + kPCInc);
            } else {
                for (MustAddressType addr = stridedBlock.first; addr < stridedBlock.second;
                     addr += stridedBlock.stride) {
                    myTSanMod->annotateMemoryWritePC(
                        pId,
                        lId,
                        addr,
                        stridedBlock.blocksize,
                        ((char*)fp) + kPCInc);
                }
            }
        }
    }

    // Function exit
    myTSanMod->annotateFuncExit();
}

//=============================
// makeBlocksActive
//=============================

template <class T>
std::pair<const T*, bool> getRequestRankElement(
    sf::contfree_safe_ptr<mustPidRequestMap<T>>& map,
    int rank,
    MustRequestType req)
{
    static T nullObj{};
    auto ret = std::make_pair(&nullObj, false);
    auto smap = slock_safe_ptr(map);
    auto smapRank = smap->find(rank);
    if (smapRank == smap->end())
        return ret;
    auto smapRankRequest = smapRank->second.find(req);
    if (smapRankRequest == smapRank->second.end())
        return ret;
    return std::make_pair(&smapRankRequest->second, true);
}

// start non-blocking communication
GTI_ANALYSIS_RETURN MpiTSanAnnotations::makeBlocksActive(
    MustParallelId pId,
    MustLocationId lId,
    const MustMemIntervalListType& preparedList,
    MustRequestType request)
{
    void* pc;
    void* fp;
    I_Request* rInfo = myReqMod->getRequest(pId, request);

    LocationInfoFp lInfo;

    int rank = myPIdMod->getInfoForId(pId).rank;
    {
        auto lInfoEntry = getRequestRankElement(requestLocation, rank, request);
        if (!lInfoEntry.second) {
            PNMPI_Service_GetReturnAddress(&pc);
            PNMPI_Service_GetFunctionAddress(&fp);
            (*xlock_safe_ptr(requestLocation))[rank][request] = lInfo =
                LocationInfoFp(pId, lId, fp, pc);
        } else
            lInfo = *lInfoEntry.first;
    }
    if (!rInfo->isPersistent()) { // record for later use
        auto x_safe_requestBlocklist = xlock_safe_ptr(requestBlocklist);
        (*x_safe_requestBlocklist)[rank][request].insert(preparedList);
    }

    // create fiber with current sync state and associate Request object with it
#ifdef ENABLE_FIBERS
    // create fiber only of not yet created for this request
    auto fiberInfo = getRequestRankElement(fiberList, rank, request);
    auto* fiber = *fiberInfo.first;
    if (!fiberInfo.second) {
        auto x_safe_fiberList = xlock_safe_ptr(fiberList);
        myTSanMod->annotateFuncEntry(lInfo.pc);
        myTSanMod->annotateFuncEntry(((char*)lInfo.fp) + kPCInc);
        fiber = myTSanMod->createFiber(0);
        assert(fiber);
        std::string fiberName = "Request from " + myLIdMod->toString(lInfo.pId, lInfo.lId);
        myTSanMod->setFiberName(fiber, fiberName.c_str());
        (*x_safe_fiberList)[rank][request] = fiber;
        myTSanMod->annotateFuncExit();
        myTSanMod->annotateFuncExit();
    }
#else
    myTSanMod->annotateHappensBefore(pId, lId, rInfo);
#endif

#ifdef ENABLE_FIBERS
    void* curFiber = myTSanMod->getCurrentFiber();
    if (fiber)
        myTSanMod->switchToFiber(fiber, 1);

#endif

    myTSanMod->annotateIgnoreSyncBegin();
    annotateFromMemIntervalList(pId, lId, preparedList, lInfo.pc, lInfo.fp);
    myTSanMod->annotateIgnoreSyncEnd();

#ifdef ENABLE_FIBERS
    if (fiber)
        myTSanMod->switchToFiber(curFiber, 1);

#endif

    return GTI_ANALYSIS_SUCCESS;
}

GTI_ANALYSIS_RETURN
MpiTSanAnnotations::makeBlocksInActive(MustParallelId pId, MustRequestType request)
{
    int rank = myPIdMod->getInfoForId(pId).rank;
    I_Request* rInfo = myReqMod->getRequest(pId, request);
    auto infoRes = getRequestRankElement(requestLocation, rank, request);
    if (!infoRes.second)
        return GTI_ANALYSIS_SUCCESS;
    const auto& info = *infoRes.first;

#ifdef ENABLE_FIBERS
    void* curFiber = myTSanMod->getCurrentFiber();

    auto fiberInfo = getRequestRankElement(fiberList, rank, request);
    void* rFiber = nullptr;
    assert(fiberInfo.second);
    if (fiberInfo.second)
        rFiber = *fiberInfo.first;
    assert(rFiber);
    if (rFiber)
        myTSanMod->switchToFiber(rFiber, 1);
#else
    myTSanMod->annotateHappensAfter(pId, info.lId, rInfo);
#endif

    const MustMemIntervalListType* blockList;
    if (rInfo->isPersistent()) {
        auto pblRes =
            getRequestRankElement(persistentBlocklist, myPIdMod->getInfoForId(pId).rank, request);
        if (!pblRes.second)
            return GTI_ANALYSIS_SUCCESS;
        blockList = pblRes.first;
    } else {
        auto pblRes =
            getRequestRankElement(requestBlocklist, myPIdMod->getInfoForId(pId).rank, request);
        if (!pblRes.second)
            return GTI_ANALYSIS_SUCCESS;
        blockList = pblRes.first;
    }

    // ignore any synchronization implied while annotating mem access (otherwise we might run
    // into false negatives due to usage of safeptr etc.)
    myTSanMod->annotateIgnoreSyncBegin();
    annotateFromMemIntervalList(info.pId, info.lId, *blockList, info.pc, info.fp);
    myTSanMod->annotateIgnoreSyncEnd();

#ifdef ENABLE_FIBERS
    // switch to current thread *with* synchronization
    // (memory accesses afterwards do *not* conflict with *this* annotated call)
    if (rFiber) {
        myTSanMod->switchToFiber(curFiber, 0);
        myTSanMod->destroyFiber(rFiber);
        (*xlock_safe_ptr(fiberList))[rank].erase(request);
    }
#endif

    if (!rInfo->isPersistent()) {

        // after deleting the blocks from activeBlocks, delete the list of bookmarks
        auto x_safe_requestLocation = xlock_safe_ptr(requestLocation);
        (*x_safe_requestLocation)[rank].erase(request);
        auto x_safe_requestBlocklist = xlock_safe_ptr(requestBlocklist);
        assert(
            (*x_safe_requestBlocklist)[rank].find(request) !=
            (*x_safe_requestBlocklist)[rank].end());
        (*x_safe_requestBlocklist)[rank].erase(request);
        assert(
            (*x_safe_requestBlocklist)[rank].find(request) ==
            (*x_safe_requestBlocklist)[rank].end());
    }

    return GTI_ANALYSIS_SUCCESS;
}

GTI_ANALYSIS_RETURN MpiTSanAnnotations::freeRequestBlocks(int rank, MustRequestType request)
{
    // delete records for location, blocklists and fiber
    {
        auto x_safe_requestLocation = xlock_safe_ptr(requestLocation);
        (*x_safe_requestLocation)[rank].erase(request);
    }

    I_Request* rInfo = myReqMod->getRequest(rank, request);
    if (rInfo->isPersistent()) {
        auto x_safe_persistentBlocklist = xlock_safe_ptr(persistentBlocklist);
        (*x_safe_persistentBlocklist)[rank].erase(request);
    } else {
        auto x_safe_requestBlocklist = xlock_safe_ptr(requestBlocklist);
        assert(
            (*x_safe_requestBlocklist)[rank].find(request) !=
            (*x_safe_requestBlocklist)[rank].end());
        (*x_safe_requestBlocklist)[rank].erase(request);
        assert(
            (*x_safe_requestBlocklist)[rank].find(request) ==
            (*x_safe_requestBlocklist)[rank].end());
#ifdef ENABLE_FIBERS
        auto fiberInfo = getRequestRankElement(fiberList, rank, request);
        if (fiberInfo.second) {
            auto rFiber = *fiberInfo.first;
            myTSanMod->destroyFiber(rFiber);
            auto x_safe_fiberList = xlock_safe_ptr(fiberList);
            (*x_safe_fiberList)[rank].erase(request);
        }
#endif
    }

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// isSendRecvOverlapped
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::isSendRecvOverlapped(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType sendbuf,
    int sendcount,
    MustDatatypeType sendtype,
    MustAddressType recvbuf,
    int recvcount,
    MustDatatypeType recvtype)
{
    return MpiTSanAnnotations::isSendRecvOverlappedN(
        pId,
        lId,
        sendbuf,
        NULL,
        0,
        &sendcount,
        1,
        &sendtype,
        1,
        recvbuf,
        NULL,
        0,
        &recvcount,
        1,
        &recvtype,
        1,
        false,
        0);
}

//=============================
// isSendRecvOverlappedN
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::isSendRecvOverlappedN(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType sendbuf,
    const int* senddispls,
    int senddisplslen,
    const int* sendcounts,
    int sendcountslen,
    const MustDatatypeType* sendtypes,
    int sendtypeslen,
    MustAddressType recvbuf,
    const int* recvdispls,
    int recvdisplslen,
    const int* recvcounts,
    int recvcountslen,
    const MustDatatypeType* recvtypes,
    int recvtypeslen,
    int hasRequest,
    MustRequestType request)
{
    return isSendRecvOverlappedNInternal<int>(
        pId,
        lId,
        sendbuf,
        senddispls,
        senddisplslen,
        sendcounts,
        sendcountslen,
        sendtypes,
        sendtypeslen,
        recvbuf,
        recvdispls,
        recvdisplslen,
        recvcounts,
        recvcountslen,
        recvtypes,
        recvtypeslen,
        hasRequest,
        request);
}

//=============================
// isSendRecvOverlappedN_uint64
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::isSendRecvOverlappedN_uint64(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType sendbuf,
    const MustAddressType* senddispls,
    int senddisplslen,
    const int* sendcounts,
    int sendcountslen,
    const MustDatatypeType* sendtypes,
    int sendtypeslen,
    MustAddressType recvbuf,
    const MustAddressType* recvdispls,
    int recvdisplslen,
    const int* recvcounts,
    int recvcountslen,
    const MustDatatypeType* recvtypes,
    int recvtypeslen,
    int hasRequest,
    MustRequestType request)
{
    return isSendRecvOverlappedNInternal<MustAddressType>(
        pId,
        lId,
        sendbuf,
        senddispls,
        senddisplslen,
        sendcounts,
        sendcountslen,
        sendtypes,
        sendtypeslen,
        recvbuf,
        recvdispls,
        recvdisplslen,
        recvcounts,
        recvcountslen,
        recvtypes,
        recvtypeslen,
        hasRequest,
        request);
}

//=============================
// isSendRecvOverlappedNInternal
//=============================
template <typename displsT>
GTI_ANALYSIS_RETURN MpiTSanAnnotations::isSendRecvOverlappedNInternal(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType sendbuf,
    const displsT* senddispls,
    int senddisplslen,
    const int* sendcounts,
    int sendcountslen,
    const MustDatatypeType* sendtypes,
    int sendtypeslen,
    MustAddressType recvbuf,
    const displsT* recvdispls,
    int recvdisplslen,
    const int* recvcounts,
    int recvcountslen,
    const MustDatatypeType* recvtypes,
    int recvtypeslen,
    int hasRequest,
    MustRequestType request)
{
    if (sendcountslen < 1 || sendtypeslen < 1 || recvcountslen < 1 || recvtypeslen < 1 ||
        senddisplslen < 0 || recvdisplslen < 0 || (senddisplslen > 0 && senddispls == NULL) ||
        (recvdisplslen > 0 && recvdispls == NULL)) {
        std::cout
            << "Implementation error: incorrect call of MpiTSanAnnotations:isSendRecvOverlappedN!"
            << std::endl;
        return GTI_ANALYSIS_SUCCESS;
    }
    if (!hasRequest) {
        request = 0;
    }

    MustMemIntervalListType iList, tList;
    displsT displacement = 0;
    int count = sendcounts[0];
    MustDatatypeType type = sendtypes[0];

    I_Datatype* typeinfo = myDatMod->getDatatype(pId, type);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    MustAddressType extent = typeinfo->getExtent();

    for (int i = 0; i < sendcountslen; ++i) {
        if (senddisplslen > 1) {
            displacement = senddispls[i];
        }
        if (sendcountslen > 1) {
            count = sendcounts[i];
        }
        if (sendtypeslen > 1) {
            type = sendtypes[i];
            typeinfo = myDatMod->getDatatype(pId, type);
            if (typeinfo == NULL) {
                return GTI_ANALYSIS_SUCCESS;
            }
        } else {
            displacement *= extent;
        }
        tList = calcIntervalList(typeinfo, sendbuf + displacement, count, (MustRequestType)0, true);
        iList.insert(tList.begin(), tList.end());
    }
    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }

    for (int i = 0; i < recvcountslen; ++i) {
        if (recvdisplslen > 1) {
            displacement = recvdispls[i];
        }
        if (recvcountslen > 1) {
            count = recvcounts[i];
        }
        if (recvtypeslen > 1) {
            type = recvtypes[i];
            typeinfo = myDatMod->getDatatype(pId, type);
            if (typeinfo == NULL) {
                return GTI_ANALYSIS_SUCCESS;
            }
        } else {
            displacement *= extent;
        }
        tList =
            calcIntervalList(typeinfo, recvbuf + displacement, count, (MustRequestType)0, false);
        iList.insert(tList.begin(), tList.end());
    }
    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// isendOverlapsRequests
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::isendOverlapsRequests(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count,
    int hasRequest,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }

    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (!hasRequest) {
        request = 0;
    }
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }

    MustMemIntervalListType iList;
    iList = calcIntervalList(typeinfo, buffer, count, request, true);
    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// sendOverlapsRequests
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::sendOverlapsRequests(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }
    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }

    MustMemIntervalListType iList =
        calcIntervalList(typeinfo, buffer, count, (MustRequestType)0, true);
    annotateFromMemIntervalList(pId, lId, iList);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// sendOverlapcheckCounts
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::sendOverlapcheckCounts(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const int displs[],
    const int counts[],
    MustDatatypeType datatype,
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }

    MustMemIntervalListType iList, tList;
    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);

    if (typeinfo == NULL || displs == NULL || counts == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    MustAddressType extent = typeinfo->getExtent();

    if (!hasRequest) {
        request = 0;
    }

    for (int i = 0; i < commsize; ++i) {
        tList = calcIntervalList(typeinfo, buffer + displs[i] * extent, counts[i], request, true);
        iList.insert(tList.begin(), tList.end());
    }

    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// sendOverlapcheckTypes
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::sendOverlapcheckTypes(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const int displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    return sendOverlapcheckTypesInternal<
        int>(pId, lId, buffer, displs, counts, datatypes, commsize, hasRequest, request);
}

//=============================
// sendOverlapcheckTypes_uint64
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::sendOverlapcheckTypes_uint64(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const MustAddressType displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    return sendOverlapcheckTypesInternal<MustAddressType>(
        pId,
        lId,
        buffer,
        displs,
        counts,
        datatypes,
        commsize,
        hasRequest,
        request);
}

//=============================
// sendOverlapcheckTypesInternal
//=============================
template <typename displsT>
GTI_ANALYSIS_RETURN MpiTSanAnnotations::sendOverlapcheckTypesInternal(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const displsT displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }

    MustMemIntervalListType iList, tList;
    I_Datatype* typeinfo;

    if (displs == NULL || counts == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }

    if (!hasRequest) {
        request = 0;
    }

    for (int i = 0; i < commsize; ++i) {
        typeinfo = myDatMod->getDatatype(pId, datatypes[i]);
        if (typeinfo == NULL) {
            return GTI_ANALYSIS_SUCCESS;
        }
        tList = calcIntervalList(typeinfo, buffer + displs[i], counts[i], request, true);
        iList.insert(tList.begin(), tList.end());
    }

    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// irecvOverlapsRequests
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::irecvOverlapsRequests(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count,
    int hasRequest,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }

    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }

    MustMemIntervalListType iList = calcIntervalList(typeinfo, buffer, count, request, false);
    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// recvOverlapsRequests
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::recvOverlapsRequests(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }
    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    MustMemIntervalListType iList =
        calcIntervalList(typeinfo, buffer, count, (MustRequestType)0, false);
    annotateFromMemIntervalList(pId, lId, iList);
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// recvOverlapcheckCounts
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::recvOverlapcheckCounts(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const int displs[],
    const int counts[],
    MustDatatypeType datatype,
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }
    MustMemIntervalListType iList, tList;
    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL || displs == NULL || counts == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    MustAddressType extent = typeinfo->getExtent();
    if (!hasRequest) {
        request = 0;
    }

    for (int i = 0; i < commsize; ++i) {
        tList = calcIntervalList(typeinfo, buffer + displs[i] * extent, counts[i], request, false);
        iList.insert(tList.begin(), tList.end());
    }

    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// recvOverlapcheckTypes
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::recvOverlapcheckTypes(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const int displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    return recvOverlapcheckTypesInternal<
        int>(pId, lId, buffer, displs, counts, datatypes, commsize, hasRequest, request);
}

//=============================
// recvOverlapcheckTypes_uint64
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::recvOverlapcheckTypes_uint64(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const MustAddressType displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    return recvOverlapcheckTypesInternal<MustAddressType>(
        pId,
        lId,
        buffer,
        displs,
        counts,
        datatypes,
        commsize,
        hasRequest,
        request);
}

//=============================
// recvOverlapcheckTypesInternal
//=============================
template <typename displsT>
GTI_ANALYSIS_RETURN MpiTSanAnnotations::recvOverlapcheckTypesInternal(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const displsT displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize,
    int hasRequest,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }
    MustMemIntervalListType iList, tList;
    I_Datatype* typeinfo;

    if (displs == NULL || counts == NULL || datatypes == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (!hasRequest) {
        request = 0;
    }

    for (int i = 0; i < commsize; ++i) {
        typeinfo = myDatMod->getDatatype(pId, datatypes[i]);
        if (typeinfo == NULL) {
            return GTI_ANALYSIS_SUCCESS;
        }
        tList = calcIntervalList(typeinfo, buffer + displs[i], counts[i], request, false);
        iList.insert(tList.begin(), tList.end());
    }

    if (hasRequest) {
        makeBlocksActive(pId, lId, iList, request);
    } else {
        annotateFromMemIntervalList(pId, lId, iList);
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// announcePSendRequest
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::announcePSendRequest(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }

    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    int rank = myPIdMod->getInfoForId(pId).rank;
    {
        auto lInfoEntry = getRequestRankElement(requestLocation, rank, request);
        if (!lInfoEntry.second) {
            void *pc, *fp;
            PNMPI_Service_GetReturnAddress(&pc);
            PNMPI_Service_GetFunctionAddress(&fp);
            (*xlock_safe_ptr(requestLocation))[rank][request] = LocationInfoFp(pId, lId, fp, pc);
        }
    }
    {
        auto x_safe_persistentBlocklist = xlock_safe_ptr(persistentBlocklist);
        (*x_safe_persistentBlocklist)[rank][request].insert(
            calcIntervalList(typeinfo, buffer, count, request, true));
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// announceSendRequest
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::announceSendRequest(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }
    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    MustMemIntervalListType iList = calcIntervalList(typeinfo, buffer, count, request, true);
    for (const auto& i : iList)
        assert(i.isSend);
    makeBlocksActive(pId, lId, iList, request);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// announcePRecvRequest
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::announcePRecvRequest(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }

    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    int rank = myPIdMod->getInfoForId(pId).rank;
    {
        auto lInfoEntry = getRequestRankElement(requestLocation, rank, request);
        if (!lInfoEntry.second) {
            void *pc, *fp;
            PNMPI_Service_GetReturnAddress(&pc);
            PNMPI_Service_GetFunctionAddress(&fp);
            (*xlock_safe_ptr(requestLocation))[rank][request] = LocationInfoFp(pId, lId, fp, pc);
        }
    }
    {
        auto x_safe_persistentBlocklist = xlock_safe_ptr(persistentBlocklist);
        (*x_safe_persistentBlocklist)[rank][request].insert(
            calcIntervalList(typeinfo, buffer, count, request, false));
    }
    //    (*xlock_safe_ptr(persistentBlocklist))[pId][request] = calcIntervalList(typeinfo, buffer,
    //    count, request, false);
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// freeRequest
//=============================
GTI_ANALYSIS_RETURN
MpiTSanAnnotations::freeRequest(MustParallelId pId, MustLocationId lId, MustRequestType request)
{
    freeRequestBlocks(myPIdMod->getInfoForId(pId).rank, request);
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// announceRecvRequest
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::announceRecvRequest(
    MustParallelId pId,
    MustLocationId lId,
    MustDatatypeType datatype,
    MustAddressType buffer,
    int count,
    MustRequestType request)
{
    if (buffer == MUST_IN_PLACE) {
        return GTI_ANALYSIS_SUCCESS;
    }
    if (buffer == MUST_BOTTOM) {
        buffer = 0;
    }
    I_Datatype* typeinfo = myDatMod->getDatatype(pId, datatype);
    if (typeinfo == NULL) {
        return GTI_ANALYSIS_SUCCESS;
    }
    MustMemIntervalListType iList = calcIntervalList(typeinfo, buffer, count, request, false);
    for (const auto& i : iList)
        assert(!i.isSend);
    makeBlocksActive(pId, lId, iList, request);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// startPRequest
//=============================
GTI_ANALYSIS_RETURN
MpiTSanAnnotations::startPRequest(MustParallelId pId, MustLocationId lId, MustRequestType request)
{
    auto pblRes =
        getRequestRankElement(persistentBlocklist, myPIdMod->getInfoForId(pId).rank, request);
    if (!pblRes.second)
        return GTI_ANALYSIS_SUCCESS;

    makeBlocksActive(pId, lId, *pblRes.first, request);

    // the same as: makeBlocksActive(pId, lId, persistentBlocklist[rank][request], request);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// startPRequestArray
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::startPRequestArray(
    MustParallelId pId,
    MustLocationId lId,
    MustRequestType* requests,
    int count)
{
    for (int i = 0; i < count; ++i) {
        startPRequest(pId, lId, requests[i]);
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// finishRequest
//=============================
GTI_ANALYSIS_RETURN
MpiTSanAnnotations::finishRequest(MustParallelId pId, MustLocationId lId, MustRequestType request)
{
    makeBlocksInActive(pId, request);
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// finishRequests
//=============================
GTI_ANALYSIS_RETURN MpiTSanAnnotations::finishRequests(
    MustParallelId pId,
    MustLocationId lId,
    MustRequestType* request,
    int count)
{
    for (int i = 0; i < count; ++i) {
        makeBlocksInActive(pId, request[i]);
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// calcIntervalList
//=============================
MustMemIntervalListType MpiTSanAnnotations::calcIntervalList(
    I_Datatype* typeinfo,
    MustAddressType buffer,
    int count,
    MustRequestType request,
    bool isSend)
{
    { // get the blocklist for the datatype and repeate as requested (count)
        BlockInfo& blockInfo = typeinfo->getBlockInfo();
        MustAddressType extent = typeinfo->getExtent();
        MustAddressType size = typeinfo->getSize();
        return buildMemIntervallist(
            blockInfo,
            extent,
            size,
            buffer,
            request,
            isSend,
            typeinfo,
            count,
            buffer);
    }

    //    return ret;
}
