/*
 * Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file MpiTypeArt.cpp
 * 	@see MpiTypeArt.
 *
 * @author Joachim Protze (RWTH Aachen), Alexander Hueck (TU Darmstadt)
 */

#include "GtiMacros.h"
#include "MpiTypeArt.h"
#include "MustEnums.h"
#include "MustDefines.h"

#include <algorithm>
#include <iostream>
#include <optional>
#include <cstddef>
#include <sstream>
#include <string>

// TypeArt runtime interface
#include <RuntimeInterface.h>
#include <TypeInterface.h>

using namespace must;

mGET_INSTANCE_FUNCTION(MpiTypeArt)
mFREE_INSTANCE_FUNCTION(MpiTypeArt)
mPNMPI_REGISTRATIONPOINT_FUNCTION(MpiTypeArt)

//=============================
// Constructor
//=============================
MpiTypeArt::MpiTypeArt(const char* instanceName)
    : gti::ModuleBase<MpiTypeArt, I_MpiTypeArt>(instanceName)
{
    // create sub modules
    std::vector<I_Module*> subModInstances;
    subModInstances = createSubModuleInstances();

    // handle sub modules
#define NUM_SUBMODULES 5
    if (subModInstances.size() < NUM_SUBMODULES) {
        std::cerr << "Module does not have enough sub modules, check its analysis specification! ("
                  << __FILE__ << "@" << __LINE__ << ")" << std::endl;
        assert(0);
    }
    if (subModInstances.size() > NUM_SUBMODULES) {
        for (std::vector<I_Module*>::size_type i = NUM_SUBMODULES; i < subModInstances.size();
             ++i) {
            destroySubModuleInstance(subModInstances[i]);
        }
    }

    myPIdMod = (I_ParallelIdAnalysis*)subModInstances[0];
    myLIdMod = (I_LocationAnalysis*)subModInstances[1];
    myLogger = (I_CreateMessage*)subModInstances[2];
    myArgMod = (I_ArgumentAnalysis*)subModInstances[3];
    myDatMod = (I_DatatypeTrack*)subModInstances[4];
}

//=============================
// Destructor
//=============================
MpiTypeArt::~MpiTypeArt(void)
{
    if (myPIdMod)
        destroySubModuleInstance((I_Module*)myPIdMod);
    myPIdMod = NULL;

    if (myLIdMod)
        destroySubModuleInstance((I_Module*)myLIdMod);
    myLIdMod = NULL;

    if (myLogger)
        destroySubModuleInstance((I_Module*)myLogger);
    myLogger = NULL;

    if (myArgMod)
        destroySubModuleInstance((I_Module*)myArgMod);
    myArgMod = NULL;

    if (myDatMod)
        destroySubModuleInstance((I_Module*)myDatMod);
    myDatMod = NULL;
}

namespace helper
{

inline bool isCompIntSize(MustMpiDatatypePredefined mpi_type, size_t size)
{
    switch (size) {
    case 1:
        return mpi_type == MUST_MPI_INT8_T;
    case 2:
        return mpi_type == MUST_MPI_INT16_T;
    case 4:
        return mpi_type == MUST_MPI_INT32_T;
    case 8:
        return mpi_type == MUST_MPI_INT64_T;
    default:
        break;
    }
    return false;
}

inline bool isCompUIntSize(MustMpiDatatypePredefined mpi_type, size_t size)
{
    switch (size) {
    case 1:
        return mpi_type == MUST_MPI_UINT8_T;
    case 2:
        return mpi_type == MUST_MPI_UINT16_T;
    case 4:
        return mpi_type == MUST_MPI_UINT32_T;
    case 8:
        return mpi_type == MUST_MPI_UINT64_T;
    default:
        break;
    }
    return false;
}

bool isCompatible(MustMpiDatatypePredefined mpi_type, int typeart_type_id)
{
    if (mpi_type == MUST_MPI_BYTE) {
        return true;
    }
    switch (typeart_type_id) {
    case TYPEART_BOOL:
        return mpi_type == MUST_MPI_C_BOOL || mpi_type == MUST_MPI_CXX_BOOL;
    case TYPEART_CHAR_8: // int8_t -> typedef for signed char
        return mpi_type == MUST_MPI_CHAR || mpi_type == MUST_MPI_SIGNED_CHAR ||
               mpi_type == MUST_MPI_PACKED || isCompIntSize(mpi_type, 1);
    case TYPEART_INT_8:
        return mpi_type == MUST_MPI_CHAR || isCompIntSize(mpi_type, 1);
    case TYPEART_INT_16:
        return mpi_type == MUST_MPI_SHORT || isCompIntSize(mpi_type, 2);
    case TYPEART_INT_32:
        return mpi_type == MUST_MPI_INT || isCompIntSize(mpi_type, 4);
    case TYPEART_INT_64:
        return mpi_type == MUST_MPI_LONG_LONG || mpi_type == MUST_MPI_LONG ||
               mpi_type == MUST_MPI_LONG_LONG_INT || isCompIntSize(mpi_type, 8);
    case TYPEART_FLOAT_32:
        return mpi_type == MUST_MPI_FLOAT;
    case TYPEART_FLOAT_64:
        return mpi_type == MUST_MPI_DOUBLE;
    case TYPEART_FLOAT_128:
        return mpi_type == MUST_MPI_LONG_DOUBLE;
    case TYPEART_UCHAR_8: // uint8_t -> typedef for unsigned char
        return mpi_type == MUST_MPI_UNSIGNED_CHAR || isCompUIntSize(mpi_type, 1);
    case TYPEART_UINT_8:
        return mpi_type == MUST_MPI_UNSIGNED_CHAR || isCompUIntSize(mpi_type, 1);
    case TYPEART_UINT_16:
        return mpi_type == MUST_MPI_UNSIGNED_SHORT || isCompUIntSize(mpi_type, 2);
    case TYPEART_UINT_32:
        return mpi_type == MUST_MPI_UNSIGNED || isCompUIntSize(mpi_type, 4);
    case TYPEART_UINT_64:
        return mpi_type == MUST_MPI_UNSIGNED_LONG_LONG || mpi_type == MUST_MPI_UNSIGNED_LONG ||
               isCompUIntSize(mpi_type, 8);
    case TYPEART_COMPLEX_64:
        return mpi_type == MUST_MPI_C_COMPLEX || mpi_type == MUST_MPI_C_FLOAT_COMPLEX ||
               mpi_type == MUST_MPI_CXX_FLOAT_COMPLEX;
    case TYPEART_COMPLEX_128:
        return mpi_type == MUST_MPI_C_DOUBLE_COMPLEX || mpi_type == MUST_MPI_CXX_DOUBLE_COMPLEX;
    case TYPEART_COMPLEX_256:
        return mpi_type == MUST_MPI_C_LONG_DOUBLE_COMPLEX ||
               mpi_type == MUST_MPI_CXX_LONG_DOUBLE_COMPLEX;
    case TYPEART_WCHAR:
        return mpi_type == MUST_MPI_WCHAR;
    default:
        break;
    }
    return false;
}
inline std::string combiner_name_for(const I_Datatype* const must_type_info)
{
    switch (must_type_info->getDatatypeClass()) {
    case MUST_TYPE_BASE:
        return "MUST_TYPE_BASE";
    case MUST_TYPE_CONTIGUOUS:
        return "MUST_type_contiguous";
    case MUST_TYPE_VECTOR:
        return "MUST_type_vector";
    case MUST_TYPE_HVECTOR:
        return "MUST_type_hvector";
    case MUST_TYPE_INDEXED:
        return "MUST_type_indexed";
    case MUST_TYPE_HINDEXED:
        return "MUST_type_hindexed";
    case MUST_TYPE_INDEXED_BLOCK:
        return "MUST_type_indexed_block";
    case MUST_TYPE_STRUCT:
        return "MUST_type_struct";
    case MUST_TYPE_RESIZED:
        return "MUST_type_resized";
    case MUST_TYPE_SUBARRAY:
        return "MUST_type_subarray";
    case MUST_TYPE_DARRAY:
        return "MUST_type_darray";
    case MUST_TYPE_UNKNOWN:
        return "MUST_type_unknown";
    }
    return "MUST_unknown_combiner";
}

inline std::string get_mpi_name_for(I_Datatype* const must_type_info)
{
    if (must_type_info->isPredefined()) {
        return must_type_info->getPredefinedName();
    }
    return combiner_name_for(must_type_info);
}

inline size_t get_typeart_vec_size_multiplier(int typeart_type_info)
{
    if (typeart_is_vector_type(typeart_type_info)) {
        typeart_struct_layout struct_layout;
        if (typeart_resolve_type_id(typeart_type_info, &struct_layout) == TYPEART_OK) {
            return struct_layout.count[0];
        }
    }
    return 1UL;
}

} // namespace helper

namespace typecheck
{

struct MUSTErrorMsg {
    MustMessageIdNames message_id{MUST_INFO_UNIMPLEMENTED_FEATURE};
    std::string message;
};

template <typename T>
struct Expected {
    std::optional<T> result{};
    std::optional<MUSTErrorMsg> error{};

    [[nodiscard]] bool has_error() const { return error.has_value(); }

    [[nodiscard]] static Expected<T> success(T&& result) { return Expected<T>{std::move(result)}; }
    [[nodiscard]] static Expected<T> fail(const MUSTErrorMsg& result)
    {
        return Expected<T>{{}, result};
    }
};

using AnalysisResult = Expected<bool>;
struct Buffer;
struct Datatype;
using BufferResult = Expected<Buffer>;
using DatatypeResult = Expected<Datatype>;

struct Buffer {
    typeart_type_info type;
    size_t type_extent;
    size_t offset{0};

    size_t get_count() const { return type.count; }
    size_t get_type_id() const { return type.type_id; }
    size_t get_offset() const { return offset; }

    size_t get_buffer_size() const { return type.count * type_extent; }
    size_t get_type_extent() const { return type_extent; }

    std::string get_type_name() const { return typeart_get_type_name(type.type_id); }

    bool is_struct() const { return typeart_is_struct_type(type.type_id); }

    std::optional<typeart_struct_layout> get_struct_layout() const
    {
        typeart_struct_layout struct_layout;
        auto result = typeart_resolve_type_id(type.type_id, &struct_layout);
        if (TYPEART_OK != result) {
            return std::nullopt;
        }
        return struct_layout;
    }

    [[nodiscard]] std::optional<Buffer> get_containing_struct() const
    {
        size_t offset{0};
        typeart_base_type_info_t containing;
        const auto status = typeart_get_containing_type(type, &containing, &offset);

        if (status != TYPEART_OK || !typeart_is_struct_type(containing.type_id)) {
            return std::nullopt;
        }
        const auto typeart_byte_size = typeart_get_type_size(containing.type_id);

        return Buffer{
            {containing.address, containing.type_id, containing.count},
            typeart_byte_size,
            offset};
    }

    [[nodiscard]] static BufferResult create(const void* ptr)
    {
        assert(ptr != nullptr && "Pointer should be non-null");

        typeart_type_info_t info;
        typeart_status status = typeart_get_type(ptr, &info);

        if (status != TYPEART_OK) {
            if (status == TYPEART_BAD_ALIGNMENT) {
                return BufferResult::fail(
                    {MUST_ERROR_TYPEMATCH_ALIGNMENT,
                     "Buffer address does not align with the underlying type."});
            } else if (status == TYPEART_UNKNOWN_ADDRESS) {
                return BufferResult::fail(
                    {MUST_INFO_UNIMPLEMENTED_FEATURE, "No buffer allocated at given address."});
            } else if (status == TYPEART_INVALID_ID) {
                return BufferResult::fail(
                    {MUST_INFO_UNIMPLEMENTED_FEATURE,
                     "Buffer has invalid TypeART type id at given address."});
            }
        }

        const auto typeart_byte_size = typeart_get_type_size(info.type_id);

        return BufferResult::success(Buffer{info, typeart_byte_size});
    }
};

struct Datatype {
    I_Datatype* must_type_info;
    I_Datatype* base_type;
    MustMpiDatatypeClass combiner;
    I_DatatypeTrack* datatype_track;
    // size_t num_elements;
    size_t mpi_count;
    long byte_size;
    const void* start_address;

    size_t num_elements() const { return must_type_info->getTypesig().front().first * mpi_count; }

    [[nodiscard]] static DatatypeResult create(
        I_Datatype* const must_type_info,
        MustAddressType buffer,
        int count,
        I_DatatypeTrack* datatype_tracker)
    {
        assert(count > 0 && "MPI count should be positive number.");

        const auto mpi_combiner = must_type_info->getDatatypeClass();
        // const bool simple_array = must_type_info->isSimpleArray();
        // if (!simple_array && mpi_combiner) {
        //     return DatatypeResult::fail(
        //         {MUST_INFO_UNIMPLEMENTED_FEATURE,
        //          "Not yet implemented: MPI datatype is not an array-like type of structs or "
        //          "base-types."});
        // }

        switch (mpi_combiner) {
        case MUST_TYPE_SUBARRAY:
        // case MUST_TYPE_HINDEXED:
        case MUST_TYPE_DARRAY:
            return DatatypeResult::fail(
                {MUST_INFO_UNIMPLEMENTED_FEATURE,
                 "Not yet implemented: Type checking combiner " +
                     helper::get_mpi_name_for(must_type_info)});
        default:
            break;
        };

        const void* startAddr = reinterpret_cast<const void*>(
            (MUST_BOTTOM == buffer) ? must_type_info->getLb() : (buffer + must_type_info->getLb()));

        const auto getBasicType = [](I_Datatype* info) -> I_Datatype* {
            while (info->getDatatypeClass() != MUST_TYPE_BASE &&
                   info->getDatatypeClass() != MUST_TYPE_STRUCT) {
                info = info->getReferencedTypes().front();
            }
            return info;
        };

        I_Datatype* baseType = getBasicType(must_type_info);

        const auto mpi_size_multiplier = [&]() {
            const auto mpi_combiner = must_type_info->getDatatypeClass();
            switch (mpi_combiner) {
            case MUST_TYPE_CONTIGUOUS:
            case MUST_TYPE_VECTOR:
            case MUST_TYPE_HVECTOR:
            case MUST_TYPE_SUBARRAY:
            case MUST_TYPE_INDEXED_BLOCK:
            case MUST_TYPE_INDEXED:
            case MUST_TYPE_HINDEXED:
                return must_type_info->getTrueExtent();
            default:
                return baseType->getSize();
            }
        }() * count;

        return DatatypeResult::success(Datatype{
            must_type_info,
            baseType,
            mpi_combiner,
            datatype_tracker,
            static_cast<size_t>(count),
            mpi_size_multiplier,
            startAddr});
    }
};
[[nodiscard]] AnalysisResult is_typemap_compatible_with_offset(
    const MustTypemapType& type_map,
    const Datatype& must_datatype,
    const typeart_struct_layout& struct_layout,
    const size_t constant_base_offset = 0)
{
    // TODO To handle type_map of MPI_BYTE's, we might need to iterate (starting at
    // base_offset) over each struct member (as opposed to typemap entry), and compare it to the
    // typemap entry w.r.t. type compatibility and offset (struct vs. typemap)
    for (const auto& [typemap_type, typemap_offset] : type_map) {
        // std::cerr << "typemap_entry{" << datatype_tracker->getPredefinedName(typemap_type) << ","
        //           << typemap_offset << "}\n";
        typeart_base_type_info_t subtype_info;
        size_t subtype_byte_offset{0};

        const auto status_typemap = typeart_get_subtype(
            &struct_layout,
            reinterpret_cast<const void*>(must_datatype.start_address),
            typemap_offset + constant_base_offset,
            &subtype_info,
            &subtype_byte_offset);

        // std::cerr << (typemap_offset + constant_base_offset) << ":" << subtype_byte_offset <<
        // "\n";

        if (status_typemap != TYPEART_OK) {
            std::stringstream stream;

            switch (status_typemap) {
            case TYPEART_BAD_OFFSET:
                stream << "Bad typemap offset for member of ";
                break;
            case TYPEART_BAD_ALIGNMENT:
                stream << "Bad offset, points to illegal position for member of ";
                break;
            default:
                stream << "Incompatible struct member type for of ";
                break;
            }

            const auto access_name = must_datatype.datatype_track->getPredefinedName(typemap_type);
            const auto base_name = helper::get_mpi_name_for(must_datatype.must_type_info);
            stream << "struct " << struct_layout.name << " with access MPI type " << access_name
                   << " at offset " << typemap_offset;
            if (access_name != base_name) {
                stream << " and base MPI type " << base_name;
            }

            return AnalysisResult::fail({MUST_ERROR_TYPEMATCH_ALIGNMENT, stream.str()});
        }

        const auto typemap_type_id = subtype_info.type_id;
        const bool is_builtin_type = typeart_is_builtin_type(typemap_type_id);

        if (!is_builtin_type || subtype_byte_offset != 0) {
            // handle the case when users send &(my_struct.my_sub_struct.double_type_member) as
            // MPI_DOUBLE etc, as my_struct and my_sub_struct and double_type_member have the same
            // address we need to be prepared to recursively check these cases:
            if (!is_builtin_type && (typemap_offset + constant_base_offset) == 0) {
                typeart_struct_layout sub_layout;
                const auto status = typeart_resolve_type_id(typemap_type_id, &sub_layout);
                if (status == TYPEART_OK) {
                    return is_typemap_compatible_with_offset(
                        type_map,
                        must_datatype,
                        sub_layout,
                        constant_base_offset);
                }
            }
            const char* sub_type_name = typeart_get_type_name(typemap_type_id);
            std::stringstream stream;
            stream << "Not yet implemented: Resolving nested struct typemap. With sub type "
                   << sub_type_name << ". Detected for struct " << struct_layout.name << ".";
            return AnalysisResult::fail({MUST_ERROR_TYPEMATCH_MISMATCH, stream.str()});
        }

        if (is_builtin_type && !helper::isCompatible(typemap_type, typemap_type_id)) {
            const char* sub_type_name = typeart_get_type_name(typemap_type_id);
            std::stringstream stream;
            stream << "Incompatible buffer element of type " << sub_type_name
                   << " (type id=" << typemap_type_id << ") of struct " << struct_layout.name
                   << " - expected "
                   << must_datatype.datatype_track->getPredefinedName(typemap_type) << " instead";

            return AnalysisResult::fail({MUST_ERROR_TYPEMATCH_MISMATCH, stream.str()});
        }
    }
    return AnalysisResult::success(true);
}

AnalysisResult
check_stride_compatibility(const Datatype& must_datatype, const Buffer& typeart_buffer)
{
    // if (must_datatype.combiner != MUST_TYPE_HVECTOR) {
    //     // Others unsupported for now
    //     return AnalysisResult::success(true);
    // }
    const auto& typemap = must_datatype.must_type_info->getTypemap();
    const auto stride_violation = std::any_of(
        std::begin(typemap),
        std::end(typemap),
        [extent = typeart_buffer.get_type_extent()](const auto& entry) {
            return (entry.second % extent) > 0;
        });

    if (stride_violation) {
        std::stringstream stream;
        stream << "Incompatible stride for buffer type " << typeart_buffer.get_type_name()
               << " for ";
        if (!must_datatype.must_type_info->isPredefined()) {
            stream << helper::get_mpi_name_for(must_datatype.must_type_info) << " with base type ";
        }
        stream << must_datatype.base_type->getPredefinedName();
        return AnalysisResult::fail({MUST_ERROR_TYPEMATCH_MISMATCH, stream.str()});
    }
    return AnalysisResult::success(true);
}

AnalysisResult check_type_length_error(
    const Datatype& must_type,
    const Buffer& typeart_buffer,
    size_t typeart_byte_buf_count)
{
    const size_t mpi_byte_transfer_size = must_type.byte_size;
    if (mpi_byte_transfer_size > typeart_byte_buf_count) {
        std::stringstream stream;
        stream << "Buffer too small: Transfer of type [" << must_type.mpi_count << "x\""
               << helper::get_mpi_name_for(must_type.must_type_info) << "\"] with byte count of "
               << mpi_byte_transfer_size << " longer than buffer argument of type ["
               << typeart_buffer.get_count() << "x\"" << typeart_buffer.get_type_name()
               << "\"] with byte count of " << typeart_byte_buf_count << ".";
        return AnalysisResult::fail({MUST_ERROR_TYPEMATCH_LENGTH, stream.str()});
    }
    return AnalysisResult::success(true);
};

} // namespace typecheck

//=============================
// checkSendOrRecv
//=============================
GTI_ANALYSIS_RETURN
MpiTypeArt::checkSendOrRecv(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    MustDatatypeType datatype,
    int count)
{
    using typecheck::Buffer;
    using typecheck::Datatype;
    if (count <= 0) {
        return GTI_ANALYSIS_SUCCESS;
    }

    const auto emit = [pId, lId, this](const auto& message, const auto id) {
        myLogger->createMessage(id, pId, lId, message);
    };

    auto expected_mpi_info =
        Datatype::create(myDatMod->getDatatype(pId, datatype), buffer, count, myDatMod);
    if (expected_mpi_info.has_error() || expected_mpi_info.result->start_address == nullptr) {
        if (expected_mpi_info.has_error()) {
            emit(expected_mpi_info.error->message, expected_mpi_info.error->message_id);
        }
        return GTI_ANALYSIS_SUCCESS;
    }

    const auto must_datatype = expected_mpi_info.result.value();

    auto expected_typeart_buffer = Buffer::create(must_datatype.start_address);
    if (expected_typeart_buffer.has_error()) {
        emit(expected_typeart_buffer.error->message, expected_typeart_buffer.error->message_id);
        return GTI_ANALYSIS_SUCCESS;
    }

    const auto typeart_buffer = expected_typeart_buffer.result.value();
    const auto typeart_containing_struct_buffer = typeart_buffer.get_containing_struct();
    const bool is_contained_struct_buffer = typeart_containing_struct_buffer.has_value();
    const bool is_struct = typeart_buffer.is_struct();

    auto* baseType = must_datatype.base_type;
    if (baseType->isPredefined()) {
        if (baseType->getPredefinedInfo() == MUST_MPI_BYTE) {
            // the only thing we can do is comparing the size
            const auto result = check_type_length_error(
                must_datatype,
                typeart_buffer,
                typeart_buffer.get_buffer_size());
            if (result.has_error()) {
                if (is_struct || is_contained_struct_buffer) {
                    // TODO need to adapt is_typemap_compatible to handle checking bytes against
                    // each struct member, and then remove this.
                    std::stringstream stream;
                    stream << "Not yet implemented: Checking MPI_BYTE against (multiple) struct "
                              "members."
                           << " Dependent error: " << result.error->message;
                    emit(stream.str(), result.error->message_id);
                    return GTI_ANALYSIS_SUCCESS;
                }
                emit(result.error->message, result.error->message_id);
            }
            return GTI_ANALYSIS_SUCCESS;
        }

        if (!is_struct && !is_contained_struct_buffer) {
            if (helper::isCompatible(baseType->getPredefinedInfo(), typeart_buffer.get_type_id())) {
                const auto result = check_type_length_error(
                    must_datatype,
                    typeart_buffer,
                    typeart_buffer.get_buffer_size());
                if (result.has_error()) {
                    emit(result.error->message, result.error->message_id);
                }
            } else {
                const auto createIncompatibilityMessage = [&](const Buffer& type) {
                    std::stringstream stream;
                    stream << "Incompatible buffer of type " << type.get_type_name()
                           << " (type id=" << type.get_type_id() << ") - expected ";
                    if (!must_datatype.must_type_info->isPredefined()) {
                        stream << helper::get_mpi_name_for(must_datatype.must_type_info)
                               << " with base type ";
                    }
                    stream << baseType->getPredefinedName() << " instead";
                    emit(stream.str(), MUST_ERROR_TYPEMATCH_MISMATCH);
                };
                createIncompatibilityMessage(typeart_buffer);
            }

            // if (pId == 0) {
            //     std::cerr << "\n";
            //     must_datatype.must_type_info->printTypemapString(std::cerr);
            //     std::cerr << "\n";
            // }
            const auto stride_result =
                typecheck::check_stride_compatibility(must_datatype, typeart_buffer);
            if (stride_result.has_error()) {
                emit(stride_result.error->message, stride_result.error->message_id);
            }

            return GTI_ANALYSIS_SUCCESS;
        }
    }

    if (!is_struct && !is_contained_struct_buffer) {
        emit("Distributed struct not yet implemented.", MUST_INFO_UNIMPLEMENTED_FEATURE);
        return GTI_ANALYSIS_SUCCESS;
    }

    // Starting here, we have a struct to check:

    size_t base_offset_containing{0};
    size_t typeart_buffer_count = typeart_buffer.get_count();
    if (is_contained_struct_buffer) {
        typeart_buffer_count = typeart_containing_struct_buffer->get_count();
        base_offset_containing = typeart_containing_struct_buffer->get_offset();
    }

    const auto struct_layout = is_struct
                                   ? typeart_buffer.get_struct_layout().value()
                                   : typeart_containing_struct_buffer->get_struct_layout().value();

    // Compare size of allocation:
    const auto typeart_byte_buf_count = typeart_buffer_count * struct_layout.extent;
    const auto type_length_check_result = check_type_length_error(
        must_datatype,
        (is_struct ? typeart_buffer : typeart_containing_struct_buffer.value()),
        typeart_byte_buf_count);

    if (type_length_check_result.has_error()) {
        std::stringstream stream;
        stream << type_length_check_result.error->message;

        const auto typemap_analysis_result = typecheck::is_typemap_compatible_with_offset(
            must_datatype.base_type->getTypemap(),
            must_datatype,
            struct_layout,
            base_offset_containing);

        if (typemap_analysis_result.has_error()) {
            stream << " Dependent error: " << typemap_analysis_result.error->message;
        }

        emit(stream.str(), type_length_check_result.error->message_id);

        return GTI_ANALYSIS_SUCCESS;
    }

    if (baseType->getDatatypeClass() == MUST_TYPE_STRUCT) {
        // For now the MPI type and the user struct type need to match exactly
        // w.r.t. extent:
        const auto analysis_result = typecheck::is_typemap_compatible_with_offset(
            must_datatype.base_type->getTypemap(),
            must_datatype,
            struct_layout,
            base_offset_containing);

        if (static_cast<size_t>(baseType->getExtent()) != struct_layout.extent) {
            std::stringstream stream;
            stream << "Not yet implemented: Extent of struct differs from MPI extent.";

            if (analysis_result.has_error()) {
                stream << " Dependent error: " << analysis_result.error->message;
            }

            emit(stream.str(), MUST_INFO_UNIMPLEMENTED_FEATURE);
        } else if (analysis_result.has_error()) {
            emit(analysis_result.error->message, analysis_result.error->message_id);
        }

        return GTI_ANALYSIS_SUCCESS;
    }

    const auto type_map_to_check = [&](const Datatype& must_datatype) {
        // must_datatype.must_type_info->printTypemapString(std::cerr);
        const auto& type_map = must_datatype.must_type_info->getTypemap();
        if (!must_datatype.must_type_info->isPredefined()) {
            return type_map;
        }
        // Build a typemap for built-in MPI type & transfer count:
        MustTypemapType built_in_typemap;
        const auto num_elements = must_datatype.num_elements();
        const auto offset_for_type = baseType->getSize();
        const auto datatype = type_map.front().first;
        for (size_t i = 0; i < num_elements; ++i) {
            built_in_typemap.emplace_back(datatype, i * offset_for_type);
        }
        return built_in_typemap;
    };

    const auto analysis_result = typecheck::is_typemap_compatible_with_offset(
        type_map_to_check(must_datatype),
        must_datatype,
        struct_layout,
        base_offset_containing);

    if (analysis_result.has_error()) {
        emit(analysis_result.error->message, analysis_result.error->message_id);
    }

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// checkSendOrRecvCounts
//=============================
GTI_ANALYSIS_RETURN
MpiTypeArt::checkSendOrRecvCounts(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const int displs[],
    const int counts[],
    MustDatatypeType datatype,
    int commsize)
{
    GTI_ANALYSIS_RETURN ret;
    for (int i = 0; i < commsize; i++) {
        ret = checkSendOrRecv(pId, lId, buffer + displs[i], datatype, counts[i]);
        if (GTI_ANALYSIS_SUCCESS != ret)
            return ret;
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// checkSendOrRecvTypes
//=============================
GTI_ANALYSIS_RETURN MpiTypeArt::checkSendOrRecvTypes(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const int displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize)
{
    GTI_ANALYSIS_RETURN ret;
    for (int i = 0; i < commsize; i++) {
        ret = checkSendOrRecv(pId, lId, buffer + displs[i], datatypes[i], counts[i]);
        if (GTI_ANALYSIS_SUCCESS != ret)
            return ret;
    }
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// checkSendOrRecvTypes_uint64
//=============================
GTI_ANALYSIS_RETURN MpiTypeArt::checkSendOrRecvTypes_uint64(
    MustParallelId pId,
    MustLocationId lId,
    MustAddressType buffer,
    const MustAddressType displs[],
    const int counts[],
    const MustDatatypeType datatypes[],
    int commsize)
{
    GTI_ANALYSIS_RETURN ret;
    for (int i = 0; i < commsize; i++) {
        ret = checkSendOrRecv(pId, lId, buffer + displs[i], datatypes[i], counts[i]);
        if (GTI_ANALYSIS_SUCCESS != ret)
            return ret;
    }
    return GTI_ANALYSIS_SUCCESS;
}
