/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file File.cpp
 *       @see File.cpp
 */

#include "File.h"

using namespace must;

//=============================
// Constructor (invalid)
//=============================
File::File()
    : HandleInfoBase("File"), myIsNull(true), myContextId(0), myComm(NULL), myCreationPId(0),
      myCreationLId(0), myPReachableBegin(NULL), myPReachableEnd(NULL)
{
    // Nothing to do
}

//=============================
// Constructor (with pointers to reachable rank interval)
//=============================
File::File(int* pReachableBegin, int* pReachableEnd)
    : HandleInfoBase("File"), myIsNull(true), myContextId(0), myComm(NULL), myCreationPId(0),
      myCreationLId(0), myPReachableBegin(pReachableBegin), myPReachableEnd(pReachableEnd)
{
    // Nothing to do
}

//=============================
// Destructor
//=============================
File::~File()
{
    if (myComm)
        myComm->erase();
    myComm = NULL;
}

//=============================
// isNull
//=============================
bool File::isNull(void) { return myIsNull; }

//=============================
// getComm
//=============================
I_CommPersistent* File::getComm(void) { return myComm; }

//=============================
// getCommHandle
//=============================
MustCommType File::getCommHandle(void) { return myCommHandle; }

//=============================
// getContextId
//=============================
unsigned long long File::getContextId(void) { return myContextId; }

//=============================
// getCreationPId
//=============================
MustParallelId File::getCreationPId(void) { return myCreationPId; }

//=============================
// getCreationLId
//=============================
MustLocationId File::getCreationLId(void) { return myCreationLId; }

void File::setTypeIdEtype(MustDatatypeType typeIdEtype) { this->typeIdEtype = typeIdEtype; }

void File::setTypeIdFiletype(MustDatatypeType typeIdFiletype)
{
    this->typeIdFiletype = typeIdFiletype;
}

void File::setInitialDisplacement(size_t initialDisplacement)
{
    this->initialDisplacement = initialDisplacement;
}

MustDatatypeType File::getTypeIdEtype(void) { return this->typeIdEtype; }

MustDatatypeType File::getTypeIdFiletype(void) { return this->typeIdFiletype; }

size_t File::getInitialDiscplacement(void) { return this->initialDisplacement; }

//=============================
// compareFiles
//=============================
bool File::compareFiles(I_File* other)
{
    if (other == NULL)
        return false;

    // files should have the same communicator and the same context id
    return *getComm() == *(other->getComm()) && getContextId() == other->getContextId();
}

//=============================
// operator ==
//=============================
bool File::operator==(I_File& other)
{
    if (this == &other)
        return true;

    return compareFiles(&other);
}

//=============================
// operator !=
//=============================
bool File::operator!=(I_File& other) { return !(*this == other); }

//=============================
// printInfo
//=============================
bool File::printInfo(
    std::stringstream& out,
    std::list<std::pair<MustParallelId, MustLocationId>>* pReferences)
{
    // Is Null
    if (isNull()) {
        out << "MPI_FILE_NULL";
        return true;
    }

    pReferences->push_back(std::make_pair(getCreationPId(), getCreationLId()));

    out << "File created at reference " << pReferences->size();

    return true;
}

//=============================
// getResourceName
//=============================
std::string File::getResourceName(void) { return "File"; }

/*EOF*/
