/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file FileTrack.cpp
 *       @see MUST::FileTrack.
 */

#include "GtiMacros.h"
#include "MustEnums.h"
#include "PrefixedOstream.hpp"

#include "FileTrack.h"
#include "I_Group.h"

#include <cstdio>
#include <sstream>
#include <utility>
#include <tuple>

using namespace must;

mGET_INSTANCE_FUNCTION(FileTrack)
mFREE_INSTANCE_FUNCTION(FileTrack)
mPNMPI_REGISTRATIONPOINT_FUNCTION(FileTrack)

#define CONTEXTID_MULTIPLIER 128

//=============================
// Constructor
//=============================
FileTrack::FileTrack(const char* instanceName)
    : TrackBase<File, I_File, MustFileType, MustMpiFilePredefined, FileTrack, I_FileTrack>(
          instanceName)

{
    if (myFurtherMods.size() < 2) {
        std::cerr << "Error: the FileTrack module needs the CommTrack module as child, but it was "
                     "not available."
                  << std::endl;
        assert(0);
    }
    myCTrack = (I_CommTrack*)myFurtherMods[0];
    myDTrack = (I_DatatypeTrack*)myFurtherMods[1];
    myRTrack = (I_RequestTrack*)myFurtherMods[2];
}

//=============================
// Destructor
//=============================
FileTrack::~FileTrack()
{
    // myCTrack->notifyOfShutdown();
}

//=============================
// fileOpen
//=============================
GTI_ANALYSIS_RETURN FileTrack::fileOpen(
    MustParallelId pId,
    MustLocationId lId,
    MustCommType comm,
    int amode,
    MustFileType fh)
{
    File* newInfo = new File();
    newInfo->myIsNull = false;
    newInfo->myCommHandle = comm;
    newInfo->myComm = myCTrack->getPersistentComm(pId, comm);
    newInfo->myContextId = newInfo->myComm->getNextContextId();
    newInfo->myCreationLId = lId;
    newInfo->myCreationPId = pId;

    //==4) Register the new user file
    submitUserHandle(pId, fh, newInfo);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// fileClose
//=============================
GTI_ANALYSIS_RETURN FileTrack::fileClose(MustParallelId pId, MustLocationId lId, MustFileType fh)
{
    removeUserHandle(pId, fh);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// fileSetView
//=============================
GTI_ANALYSIS_RETURN FileTrack::fileSetView(
    MustParallelId pId,
    MustLocationId lId,
    MustFileType fh,
    int64_t displacement,
    MustDatatypeType etype,
    MustDatatypeType filetype)
{

    File* fileInfo = getHandleInfo(pId, fh);

    fileInfo->setTypeIdEtype(etype);
    fileInfo->setTypeIdFiletype(filetype);
    fileInfo->setInitialDisplacement(displacement);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// getFile
//=============================
I_File* FileTrack::getFile(MustParallelId pId, MustFileType file)
{
    return getFile(pId2Rank(pId), file);
}

//=============================
// getPersistentFile
//=============================
I_FilePersistent* FileTrack::getPersistentFile(MustParallelId pId, MustFileType file)
{
    return getPersistentFile(pId2Rank(pId), file);
}

//=============================
// getPersistentFile
//=============================
I_FilePersistent* FileTrack::getPersistentFile(int rank, MustFileType file)
{
    File* ret = getFileInfo(rank, file);
    if (ret)
        ret->incRefCount();
    return ret;
}

//=============================
// getFile
//=============================
I_File* FileTrack::getFile(int rank, MustFileType file) { return getFileInfo(rank, file); }

//=============================
// getFileInfo
//=============================
File* FileTrack::getFileInfo(MustParallelId pId, MustFileType file)
{
    return getFileInfo(pId2Rank(pId), file);
}

//=============================
// getFileInfo
//=============================
File* FileTrack::getFileInfo(int rank, MustFileType file) { return getHandleInfo(rank, file); }

//=============================
// createPredefinedInfo
//=============================
File* FileTrack::createPredefinedInfo(int value, MustFileType handle)
{
    if (handle == myNullValue) {
        return new File();
    }
    return NULL; // There should not be any other cases
}

GTI_ANALYSIS_RETURN FileTrack::addRequest(
    MustParallelId pId,
    MustLocationId lId,
    MustFileType file,
    MustRequestType request)
{
    if (auto info = getFileInfo(pId2Rank(pId), file))
        info->myOpenRequests.emplace(request, lId);
    return GTI_ANALYSIS_SUCCESS;
}

GTI_ANALYSIS_RETURN
FileTrack::removeRequest(
    MustParallelId pId,
    MustLocationId lId,
    MustFileType file,
    MustRequestType request)
{
    if (auto fInfo = getFileInfo(pId, file)) {
        auto it = fInfo->myOpenRequests.find(request);
        if (it != fInfo->myOpenRequests.end())
            fInfo->myOpenRequests.erase(it);
    }
    return GTI_ANALYSIS_SUCCESS;
}

/*EOF*/
