/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_File.h
 *       @see I_File.
 *
 *  @date 23.06.2011
 *  @author Tobias Hilbrich, Joachim Protze, Mathias Korepkat
 */

#include "MustTypes.h"
#include "MustEnums.h"
#include "BaseIds.h"

#include "I_Comm.h"
#include "I_Datatype.h"
#include "I_GroupTable.h"
#include "I_Destructable.h"

#include <unordered_map>
#include <sstream>
#include <list>

#ifndef I_File_H
#define I_File_H

namespace must
{
/**
 * Interface for storage and accessing Information
 * on a File as defined in MPI.
 */
class I_File
{
  public:
    /*
     * Basic information for a file handle.
     */
    virtual bool isNull(void) = 0; /**< True if this is MPI_COMM_NULL, isKnown=true in that case,
                                      the other pieces of information are not set. */

    // For all known and non-null communicators
    virtual unsigned long long
    getContextId(void) = 0; /**< The id that determines whether to communicator handles with equal
                               groups actually refer to the same communicator or not.*/

    /*
     * Communicator associated with the window.
     */
    virtual I_CommPersistent* getComm(void) = 0;

    /*
     * Communicator handle associated with the window.
     */
    virtual MustCommType getCommHandle(void) = 0;

    virtual void setTypeIdEtype(MustDatatypeType typeIdEtype) = 0;
    virtual void setTypeIdFiletype(MustDatatypeType typeIdFiletype) = 0;
    virtual void setInitialDisplacement(size_t initialDisplacement) = 0;
    virtual MustDatatypeType getTypeIdEtype(void) = 0;
    virtual MustDatatypeType getTypeIdFiletype(void) = 0;
    virtual size_t getInitialDiscplacement(void) = 0;
    /*
     * History information.
     */
    virtual MustParallelId getCreationPId(void) = 0; /**< For persistent comms information for call
                                                        that created the comm, otherwise not set.*/
    virtual MustLocationId getCreationLId(void) = 0; /**< For persistent comms information for call
                                                        that created the comm, otherwise not set.*/

    /**
     * Returns true if this file is equal to the given one.
     * (ToDo)
     * Returns false otherwise.
     *
     * @param other file to compare to.
     */
    virtual bool compareFiles(I_File* other) = 0;

    /**
     * Returns true if this communicator is equal to the given one.
     * (Refer to the same communication group with equal context id.)
     * Returns false otherwise. This is the 'operator version' of
     * compareComms().
     *
     * @param other communicator to compare to.
     */
    virtual bool operator==(I_File& other) = 0;

    /**
     * Returns false if this communicator is equal to the given one.
     * (Refer to the same communication group with equal context id.)
     * Returns true otherwise. This is the 'operator version' of
     * the opposite of compareComms().
     *
     * @param other communicator to compare to.
     */
    virtual bool operator!=(I_File& other) = 0;

    /**
     * Prints information for a specified comm.
     * Designed for printing in a style that suits the usage
     * of CreateMessage.
     *
     * @param out stream to use for output.
     * @param pReferences current references to which any additional references for the new handle
     * will be added.
     * @return true if successful.
     */
    virtual bool printInfo(
        std::stringstream& out,
        std::list<std::pair<MustParallelId, MustLocationId>>* pReferences) = 0;

    virtual std::unordered_map<MustRequestType, MustLocationId>& getOpenRequests(void) = 0;
}; /*class I_File*/

/**
 * Interface for storage and accessing Information
 * on a file as defined in MPI. This is the persistent
 * version of the interface. The user needs to call I_FilePersistent::erase
 * when he is finished with it.
 */
class I_FilePersistent : public I_File, public virtual I_Destructable
{
}; /*class I_FilePersistent*/

} /*namespace must*/

#endif /*I_File_H*/
