/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_FileTrack.h
 *       @see I_FileTrack.
 */

#include "I_Module.h"
#include "GtiEnums.h"

#include "MustEnums.h"
#include "MustTypes.h"
#include "BaseIds.h"
#include "I_ChannelId.h"
#include "I_GroupTrack.h"
#include "I_GroupTable.h"
#include "I_File.h"
#include "I_TrackBase.h"

#include <mpi.h>
#include <list>

#ifndef I_FILETRACK_H
#define I_FILETRACK_H

/**
 * Interface for querying information on files.
 *
 * Important: This analysis module only tracks files,
 * it provides no correctness checking. However, it tries
 * to handle incorrect actions as good as possible.
 *
 * Dependencies (in listed order):
 * - ParallelIdAnalysis
 * - LocationAnalysis
 * - GroupTrack
 */
class I_FileTrack : public gti::I_Module, public virtual must::I_TrackBase<must::I_File>
{
  public:
    /**
     * @see I_FileTrack::fileOpen.
     */
    virtual gti::GTI_ANALYSIS_RETURN fileOpen(
        MustParallelId pId,
        MustLocationId lId,
        MustCommType comm,
        int amode,
        MustFileType fh) = 0;

    virtual gti::GTI_ANALYSIS_RETURN
    fileClose(MustParallelId pId, MustLocationId lId, MustFileType fh) = 0;

    virtual gti::GTI_ANALYSIS_RETURN fileSetView(
        MustParallelId pId,
        MustLocationId lId,
        MustFileType fh,
        int64_t displacement,
        MustDatatypeType etype,
        MustDatatypeType filetype) = 0;

    virtual must::I_File* getFile(MustParallelId pId, MustFileType file) = 0;

    /** As I_CommTrack::getFile with rank instead of pid.*/
    virtual must::I_File* getFile(int rank, MustFileType file) = 0;

    virtual must::I_FilePersistent* getPersistentFile(MustParallelId pId, MustFileType file) = 0;

    /** As I_FileTrack::getPersistentFile with rank instead of pid.*/
    virtual must::I_FilePersistent* getPersistentFile(int rank, MustFileType file) = 0;

    /**
     * Returns a list of all currently known user handles.
     * Usage scenarios involve logging lost handles at finalize.
     * @return a list of pairs of the form (rank, handle id).
     */
    virtual std::list<std::pair<int, MustFileType>> getUserHandles(void) = 0;
    virtual bool isPredefined(must::I_File* type) { return false; }

    /**
     * Adds the integer (MustRequestType) values for all predefined
     * (named) handles for requests.
     */
    virtual gti::GTI_ANALYSIS_RETURN addPredefineds(
        MustParallelId pId,
        MustFileType fileNull,
        int numPredefs,
        int* predefinedIds,
        MustFileType* predefinedValues) = 0;

    virtual gti::GTI_ANALYSIS_RETURN addRequest(
        MustParallelId pId,
        MustLocationId lId,
        MustFileType file,
        MustRequestType request) = 0;

    virtual gti::GTI_ANALYSIS_RETURN removeRequest(
        MustParallelId pId,
        MustLocationId lId,
        MustFileType file,
        MustRequestType request) = 0;

}; /*class I_FileTrack*/

#endif /*I_FILETRACK_H*/
