/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_Info.h
 *       @see I_Info.
 *
 *  @date 23.06.2011
 *  @author Tobias Hilbrich, Joachim Protze, Mathias Korepkat
 */

#include "MustEnums.h"
#include "BaseIds.h"
#include <string>
#include <list>
#include <iostream>
#include <unordered_map>

#include "I_Destructable.h"

#ifndef I_INFO_H
#define I_INFO_H

namespace must
{
/**
 * Interface for storage and accessing Information
 * on a info as defined in MPI.
 */
class I_Info
{
  public:
    /*
     * Basic information for an info handle.
     */
    virtual bool isNull(void) = 0;       /**< True if this is MPI_INFO_NULL, isKnown=true in that
                                            case, the other pieces of information are not set. */
    virtual bool isPredefined(void) = 0; /**< True if this is a predefined info.*/

    /*
     * History information.
     */
    virtual MustParallelId getCreationPId(void) = 0; /**< For persistent infos information for call
                                                        that created the info, otherwise not set.*/
    virtual MustLocationId getCreationLId(void) = 0; /**< For persistent infos information for call
                                                        that created the info, otherwise not set.*/

    /**
     * For infos that are predefined and not MPI_INFO_NULL,
     * returns an enumeration that identifies the name of the
     * predefined info.
     *
     * @return value of predefined info enumeration.
     */
    virtual MustMpiInfoPredefined getPredefinedInfo(void) = 0;

    /**
     * If this is a predefined handle, returns the textual name of the
     * predefined MPI handle it represents.
     * @return name of handle.
     */
    virtual std::string getPredefinedName(void) = 0;

    /**
     * Returns the (key,val) pairs of handle.
     * @return (key,val) pairs of handle.
     */
    virtual std::unordered_map<std::string, std::string>& getKeyValPairs(void) = 0;

    /**
     * Prints information for a specified info.
     * Designed for printing in a style that suits the usage
     * of CreateMessage.
     *
     * @param out stream to use for output.
     * @param pReferences current references to which any additional references for the new handle
     * will be added.
     * @return true if successful.
     */
    virtual bool printInfo(
        std::stringstream& out,
        std::list<std::pair<MustParallelId, MustLocationId>>* pReferences) = 0;

}; /*class I_Info*/

/**
 * Interface for storage and accessing Information
 * on a info as defined in MPI. This is the persistent
 * version of the interface. The user needs to call I_InfoPersistent::erase
 * when he is finished with it.
 */
class I_InfoPersistent : public I_Info, public virtual I_Destructable
{
}; /*class I_InfoPersistent*/

} /*namespace must*/

#endif /*I_INFO_H*/
