/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_InfoTrack.h
 *       @see I_InfoTrack.
 *
 *  @date 12.05.2011
 *  @author Tobias Hilbrich
 */

#include "I_Module.h"
#include "GtiEnums.h"

#include "MustEnums.h"
#include "BaseIds.h"
#include "MustTypes.h"
#include "I_Info.h"
#include "I_TrackBase.h"

#include <list>

#ifndef I_INFOTRACK_H
#define I_INFOTRACK_H

/**
 * Interface for querying information on info.
 *
 * Important: This analysis module only tracks infos,
 * it provides no correctness checking. However, it tries
 * to handle incorrect actions as good as possible.
 *
 * Dependencies (in listed order):
 * - ParallelIdAnalysis
 * - LocationAnalysis
 */
class I_InfoTrack : public gti::I_Module, public virtual must::I_TrackBase<must::I_Info>
{
  public:
    /**
     * Creates a new user defined info.
     *
     * @param pId parallel id of context.
     * @param lId location id of context.
     * @param commute true if commutative.
     * @param newInfoHandle the newly created info.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN infoCreate(
        MustParallelId pId,
        MustLocationId lId,
        MustInfoType newInfoHandle,
        int keyValuesLen,
        char* keyValues) = 0;

    /**
     * Creates a new user defined info.
     *
     * @param pId parallel id of context.
     * @param lId location id of context.
     * @param commute true if commutative.
     * @param infoHandle the newly created info.
     * @param key the new key.
     * @param value the new value.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN infoSet(
        MustParallelId pId,
        MustLocationId lId,
        MustInfoType infoHandle,
        const char* key,
        const char* value) = 0;

    /**
     * Frees the given info.
     *
     * @param pId parallel id of context.
     * @param lId location id of context.
     * @param infoHandle to free.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    infoFree(MustParallelId pId, MustLocationId lId, MustInfoType infoHandle) = 0;

    /**
     * Returns a pointer to info information.
     * Is NULL if this is an unknown handle, note that
     * a MPI_INVALID_KEY handle returns a valid pointer though.
     *
     * Memory must not be freed and is valid until I_InfoTrack
     * receives the next event, if you need the information longer
     * query getPersistentInfo instead.
     *
     * @param pId of the context.
     * @param infoHandle to query for.
     * @return information for the given info.
     */
    virtual must::I_Info* getInfo(MustParallelId pId, MustInfoType infoHandle) = 0;

    /** As I_InfoTrack::getInfo with rank instead of pid.*/
    virtual must::I_Info* getInfo(int rank, MustInfoType infoHandle) = 0;

    /**
     * Like I_InfoTrack::getInfo, though returns a persistent information
 * that is valid until you erase it, i.e.:
 *@code
 I_InfoPersistent* infoInfo = myInfoTrack->getPersistentInfo (pId, handle);
 if (infoInfo == NULL) return;
 .... //Do something with infoInfo
 infoInfo->erase(); //Mark as not needed any longer
 *@endcode
 *
 * A reference count mechanism is used to implement this.
     *
     * @param pId of the context.
     * @param infoHandle to query for.
     * @return information for the given info.
     */
    virtual must::I_InfoPersistent*
    getPersistentInfo(MustParallelId pId, MustInfoType infoHandle) = 0;

    /** As I_InfoTrack::getPersistentInfo with rank instead of pid.*/
    virtual must::I_InfoPersistent* getPersistentInfo(int rank, MustInfoType infoHandle) = 0;

    /**
     * Adds the integer (MustInfoType) values for all predefined
     * (named) handles for infos.
     *
     * @param pId of the context.
     * @param infoNull value of MPI_INFO_NULL.
     * @param numPredefs number of predefined non null infos being sent.
     * @param predefinedIds array of value of MustMpiInfoPredefined for each predefined type,
     * array size is numPredefs.
     * @param predefinedValues array of handles for the predefined types.
     *
     */
    virtual gti::GTI_ANALYSIS_RETURN addPredefineds(
        MustParallelId pId,
        MustInfoType infoNull,
        int numPredefs,
        int* predefinedIds,
        MustInfoType* predefinedValues) = 0;

    /**
     * Returns a list of all currently known user handles.
     * Usage scenarios involve logging lost handles at finalize.
     * @return a list of pairs of the form (rank, handle id).
     */
    virtual std::list<std::pair<int, MustInfoType>> getUserHandles(void) = 0;

    /**
     * Allows other modules to notify this module of an ongoing shutdown.
     * This influcences the behavior of passing free calls across to other layers.
     */
    virtual void notifyOfShutdown(void) = 0;
    virtual bool isPredefined(must::I_Info* info) { return false; }

}; /*class I_InfoTrack*/

#endif /*I_INFOTRACK_H*/
