/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file InfoTrack.cpp
 *       @see MUST::InfoTrack.
 *
 *  @date 12.05.2011
 *  @author Mathias Korepkat, Tobias Hilbrich
 */

#include "GtiMacros.h"
#include "MustEnums.h"
#include "PrefixedOstream.hpp"

#include "InfoTrack.h"

#include <sstream>

using namespace must;

mGET_INSTANCE_FUNCTION(InfoTrack)
mFREE_INSTANCE_FUNCTION(InfoTrack)
mPNMPI_REGISTRATIONPOINT_FUNCTION(InfoTrack)

//=============================
// Constructor
//=============================
InfoTrack::InfoTrack(const char* instanceName)
    : TrackBase<Info, I_Info, MustInfoType, MustMpiInfoPredefined, InfoTrack, I_InfoTrack>(
          instanceName)
{
    // Nothing to do
}

//=============================
// Destructor
//=============================
InfoTrack::~InfoTrack()
{
    // Notify HandleInfoBase of ongoing shutdown
    HandleInfoBase::disableFreeForwardingAcross();
}

//=============================
// infoCreate
//=============================
GTI_ANALYSIS_RETURN
InfoTrack::infoCreate(
    MustParallelId pId,
    MustLocationId lId,
    MustInfoType newInfoHandle,
    int keyValuesLen,
    char* keyValues)
{
    //==Should not be known yet
    Info* newInfo = getHandleInfo(pId, newInfoHandle);
    if (newInfo) {
        if (!newInfo->myIsNull && !newInfo->myIsPredefined)
            newInfo->mpiIncRefCount();

        return GTI_ANALYSIS_SUCCESS;
    }

    //==Create the full info
    Info* info = new Info();

    info->myIsNull = false;
    info->myIsPredefined = false;
    info->myCreationPId = pId;
    info->myCreationLId = lId;

    //==There are key value pairs to be inherited
    if (!(keyValues == NULL || *keyValues == '\0')) {
        //== Store keyValues in myKeyValPairs
        int stride = MPI_MAX_INFO_KEY + MPI_MAX_INFO_VAL, nkeys = keyValuesLen / stride;

        for (int i = 0; i < nkeys; i++) {
            int offset = i * stride;
            const char* key_ptr = keyValues + offset;
            const char* val_ptr = key_ptr + MPI_MAX_INFO_KEY;

            std::string key(key_ptr);
            std::string val(val_ptr);

            info->myKeyValPairs[key] = val;
        }
    }

    //==Store the full info in the user handles
    submitUserHandle(pId, newInfoHandle, info);

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// infoSet
//=============================
GTI_ANALYSIS_RETURN
InfoTrack::infoSet(
    MustParallelId pId,
    MustLocationId lId,
    MustInfoType infoHandle,
    const char* key,
    const char* value)
{
    Info* info = getHandleInfo(pId, infoHandle);

    if (info == NULL || info->myIsNull) {
        return GTI_ANALYSIS_SUCCESS;
    }

    std::unordered_map<std::string, std::string>& keyValPairs = info->getKeyValPairs();

    keyValPairs.insert({std::string(key), std::string(value)});

    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// infoFree
//=============================
GTI_ANALYSIS_RETURN
InfoTrack::infoFree(MustParallelId pId, MustLocationId lId, MustInfoType infoHandle)
{
    removeUserHandle(pId, infoHandle);
    return GTI_ANALYSIS_SUCCESS;
}

//=============================
// getInfo
//=============================
I_Info* InfoTrack::getInfo(MustParallelId pId, MustInfoType infoHandle)
{
    return getInfo(pId2Rank(pId), infoHandle);
}

//=============================
// getInfo
//=============================
I_Info* InfoTrack::getInfo(int rank, MustInfoType infoHandle)
{
    return getHandleInfo(rank, infoHandle);
}

//=============================
// getPersistentInfo
//=============================
I_InfoPersistent* InfoTrack::getPersistentInfo(MustParallelId pId, MustInfoType infoHandle)
{
    return getPersistentInfo(pId2Rank(pId), infoHandle);
}

//=============================
// getPersistentInfo
//=============================
I_InfoPersistent* InfoTrack::getPersistentInfo(int rank, MustInfoType infoHandle)
{
    Info* info = getHandleInfo(rank, infoHandle);
    if (info)
        info->mpiIncRefCount();
    return info;
}

//=============================
// getPredefinedName
//=============================
std::string InfoTrack::getPredefinedName(MustMpiInfoPredefined predefined)
{
    switch (predefined) {
    case MUST_MPI_INFO_ENV:
        return "MPI_INFO_ENV";
    case MUST_MPI_INFO_UNKNOWN:
        return "Unknown attribute info";
    default:
        must::cerr << "Error: Unknown info enum in " << __FILE__ << ":" << __LINE__
                   << " check mapping." << std::endl;
        assert(0);
    }

    return "";
}

//=============================
// createPredefinedInfo
//=============================
Info* InfoTrack::createPredefinedInfo(int predef, MustInfoType handle)
{
    if (handle == myNullValue)
        return new Info();

    MustMpiInfoPredefined e = (MustMpiInfoPredefined)predef;
    return new Info(e, getPredefinedName(e));
}

/*EOF*/
