/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_BaseConstants.h
 *       @see I_BaseConstants.
 *
 *  @date 12.04.2011
 *  @author Mathias Korepkat
 */

#include "I_Module.h"
#include "GtiEnums.h"
#include "BaseIds.h"

#ifndef I_BASECONSTANTS_H
#define I_BASECONSTANTS_H

/**
 * Interface for providing basic MPI constants.
 * It is used to save values of MPI constants.
 * It should be invoked after (post) MPI_Init.
 *
 * Usage scenarios include comparing values -- such as
 * tags and ranks -- against MPI constants, in order to
 * determine their correctness. Usually these constants
 * are negative values, and must thus be considered when
 * determining whether some value is really invalid.
 */
class I_BaseConstants : public gti::I_Module
{
  public:
    /**
     * Gets the given constants and save them.
     * @param mpiProcNull value of MPI_PROC_NULL.
     * @param mpiAnySource value of MPI_ANY_SOURCE.
     * @param mpiAnyTag value of MPI_ANY_TAG.
     * @param mpiUndefined value of MPI_UNDEFINED.
     * @param mpiBsendOverhead value of MPI_BSEND_OVERHEAD.
     * @param mpiTagUb upper bound for tags, querried as property.
     * @param mpiMaxInfoKey maximum length for keys of info objects.
     * @param mpiMaxInfoVal maximum length for values of info objects.
     * @param mpiVersion value of major version from MPI_Get_version.
     * @param mpiSubversion value of minor version from MPI_Get_version.
     * @param mpiDistributeBlock value of MPI_DISTRIBUTE_BLOCK.
     * @param mpiDistributeCyclic value of MPI_DISTRIBUTE_CYCLIC.
     * @param mpiDistributeNone value of MPI_DISTRIBUTE_NONE.
     * @param mpiDistributeDefautDarg value of MPI_DISTRIBUTE_NONE.
     * @param mpiOrderC value of MPI_ORDER_C.
     * @param mpiOrderFortran value of MPI_ORDER_FORTRAN.
     * @param mpiBottom address of MPI_BOTTOM
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN addConstants(
        int mpiProcNull,
        int mpiAnySource,
        int mpiAnyTag,
        int mpiUndefined,
        int mpiBsendOverhead,
        int mpiTagUb,
        int mpiMaxInfoKey,
        int mpiMaxInfoVal,
        int mpiVersion,
        int mpiSubversion,
        int mpiDistributeBlock,
        int mpiDistributeCyclic,
        int mpiDistributeNone,
        int mpiDistributeDefautDarg,
        int mpiOrderC,
        int mpiOrderFortran,
        void* mpiBottom) = 0;

    /**
     * Gets the given RMA constants and saves them.
     * @param mpiModeNocheck value of MPI_MODE_NOCHECK.
     * @param mpiModeNostore value of MPI_MODE_NOSTORE.
     * @param mpiModeNoput value of MPI_MODE_NOPUT.
     * @param mpiModeNocprecede value of MPI_MODE_NOPRECEDE.
     * @param mpiModeNosucceed value of MPI_MODE_NOSUCCEED.
     * @param mpiLockExclusive value of MPI_LOCK_EXCLUSIVE.
     * @param mpiModeLockShared value of MPI_LOCK_SHARED.
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN addRMAConstants(
        int mpiModeNocheck,
        int mpiModeNostore,
        int mpiModeNoput,
        int mpiModeNoprecede,
        int mpiModeNosucceed,
        int mpiLockExclusive,
        int mpiModeLockShared) = 0;

    /**
     * Checks if an argument is equal to MPI_PROC_NULL
     * @param val value that is checked
     * @return true if val is equal to MPI_PROC_NULL
     */
    virtual bool isProcNull(int val) = 0;

    /**
     * Returns the constant for MPI_PROC_NULL.
     * @return value of MPI_PROC_NULL.
     */
    virtual int getProcNull(void) = 0;

    /**
     * Checks if an argument is equal to MPI_ANY_SOURCE.
     * @param val value that is checked
     * @return true if val is equal to MPI_ANY_SOURCE
     */
    virtual bool isAnySource(int val) = 0;

    /**
     * Returns the constant for MPI_ANY_SOURCE.
     * @return value for MPI_ANY_SOURCE.
     */
    virtual int getAnySource(void) = 0;

    /**
     * Checks if an argument is equal to MPI_ANY_TAG.
     * @param val value that is checked
     * @return true if val is equal to MPI_ANY_TAG
     */
    virtual bool isAnyTag(int val) = 0;

    /**
     * Returns the constant for MPI_ANY_TAG.
     * @return value of MPI_ANY_TAG.
     */
    virtual int getAnyTag(void) = 0;

    /**
     * Checks if an argument is equal to MPI_UNDEFINED.
     * @param val value that is checked
     * @return true if val is equal to MPI_UNDEFINED
     */
    virtual bool isUndefined(int val) = 0;

    /**
     * Returns the value of MPI_UNDEFINED.
     * @return value of MPI_UNDEFINED.
     */
    virtual int getUndefined(void) = 0;

    /**
     * Checks if an argument is equal to MPI_BSEND_OVERHEAD.
     * @param val value that is checked
     * @return true if val is equal to MPI_BSEND_OVERHEAD
     */
    virtual bool isBsendOverhead(int val) = 0;

    /**
     * Returns the value of MPI_BSEND_OVERHEAD.
     * @return value of MPI_BSEND_OVERHEAD.
     */
    virtual int getBsendOverhead(void) = 0;

    /**
     * Checks if an argument is equal to MPI_TAG_UB.
     * @param val value that is checked
     * @return true if val is equal to MPI_TAG_UB
     */
    virtual bool isTagUb(int val) = 0;

    /**
     * Returns the value of MPI_TAG_UB.
     * @return value of MPI_TAG_UB.
     */
    virtual int getTagUb(void) = 0;

    /**
     * Checks if an argument is equal to MPI_MAX_INFO_KEY.
     * @param val value that is checked
     * @return true if val is equal to MPI_MAX_INFO_KEY
     */
    virtual bool isMaxInfoKey(int val) = 0;

    /**
     * Returns the value of MPI_MAX_INFO_KEY.
     * @return value of MPI_MAX_INFO_KEY.
     */
    virtual int getMaxInfoKey(void) = 0;

    /**
     * Checks if an argument is equal to MPI_MAX_INFO_VAL.
     * @param val value that is checked
     * @return true if val is equal to MPI_MAX_INFO_VAL
     */
    virtual bool isMaxInfoVal(int val) = 0;

    /**
     * Returns the value of MPI_MAX_INFO_VAL.
     * @return value of MPI_MAX_INFO_VAL.
     */
    virtual int getMaxInfoVal(void) = 0;

    /**
     * Checks if an argument is equal to the MPI Version.
     * @param val value that is checked
     * @return true if val is equal to the major version of used MPI
     */
    virtual bool isVersion(int val) = 0;

    /**
     * Returns the value of the major version of MPI.
     * @return value of the major version of used MPI.
     */
    virtual int getVersion(void) = 0;

    /**
     * Checks if an argument is equal to the MPI Subversion.
     * @param val value that is checked
     * @return true if val is equal to the minor version of used MPI
     */
    virtual bool isSubversion(int val) = 0;

    /**
     * Returns the value of the minor version of MPI.
     * @return value of the minor version of used MPI.
     */
    virtual int getSubversion(void) = 0;

    /**
     * Checks if an argument is equal to MPI_DISTRIBUTE_BLOCK.
     * @param val value that is checked
     * @return true if val is equal MPI_DISTRIBUTE_BLOCK
     */
    virtual bool isDistributeBlock(int val) = 0;

    /**
     * Returns the value of MPI_DISTRIBUTE_BLOCK.
     * @return value of the MPI_DISTRIBUTE_BLOCK.
     */
    virtual int getDistributeBlock(void) = 0;

    /**
     * Checks if an argument is equal to MPI_DISTRIBUTE_CYCLIC.
     * @param val value that is checked
     * @return true if val is equal MPI_DISTRIBUTE_CYCLIC
     */
    virtual bool isDistributeCyclic(int val) = 0;

    /**
     * Returns the value of MPI_DISTRIBUTE_CYCLIC.
     * @return value of the MPI_DISTRIBUTE_CYCLIC.
     */
    virtual int getDistributeCyclic(void) = 0;

    /**
     * Checks if an argument is equal to MPI_DISTRIBUTE_NONE.
     * @param val value that is checked
     * @return true if val is equal MPI_DISTRIBUTE_NONE
     */
    virtual bool isDistributeNone(int val) = 0;

    /**
     * Returns the value of MPI_DISTRIBUTE_NONE.
     * @return value of the MPI_DISTRIBUTE_NONE.
     */
    virtual int getDistributeNone(void) = 0;

    /**
     * Checks if an argument is equal to MPI_DISTRIBUTE_DFLT_DARG.
     * @param val value that is checked
     * @return true if val is equal MPI_DISTRIBUTE_DFLT_DARG
     */
    virtual bool isDistributeDfltDarg(int val) = 0;

    /**
     * Returns the value of MPI_DISTRIBUTE_DFLT_DARG.
     * @return value of the MPI_DISTRIBUTE_DFLT_DARG.
     */
    virtual int getDistributeDfltDarg(void) = 0;

    /**
     * Checks if an argument is equal to MPI_ORDER_C.
     * @param val value that is checked
     * @return true if val is equal MPI_ORDER_C
     */
    virtual bool isOrderC(int val) = 0;

    /**
     * Returns the value of MPI_ORDER_C.
     * @return value of the MPI_ORDER_C.
     */
    virtual int getOrderC(void) = 0;

    /**
     * Checks if an argument is equal to MPI_ORDER_FORTRAN.
     * @param val value that is checked
     * @return true if val is equal MPI_ORDER_FORTRAN
     */
    virtual bool isOrderFortran(int val) = 0;

    /**
     * Returns the value of MPI_ORDER_FORTRAN.
     * @return value of the MPI_ORDER_FORTRAN.
     */
    virtual int getOrderFortran(void) = 0;

    /**
     * Checks if an argument is equal to the MPI_BOTTOM.
     * @param val value that is checked
     * @return true if val is equal to MPI_BOTTOM otherwise false
     */
    virtual bool isBottom(void* val) = 0;

    /**
     * Returns the value of MPI_BOTTOM.
     * @return value of MPI_BOTTOM.
     */
    virtual void* getBottom(void) = 0;

    /**
     * Returns the constant for MPI_MODE_NOCHECK.
     * @return value of MPI_MODE_NOCHECK.
     */
    virtual int getModeNocheck(void) = 0;

    /**
     * Checks if an argument is equal to MPI_MODE_NOSTORE
     * @param val value that is checked
     * @return true if val is equal to MPI_MODE_NOSTORE
     */
    virtual bool isModeNostore(int val) = 0;

    /**
     * Returns the constant for MPI_MODE_NOSTORE.
     * @return value of MPI_MODE_NOSTORE.
     */
    virtual int getModeNostore(void) = 0;

    /**
     * Checks if an argument is equal to MPI_MODE_NOPUT
     * @param val value that is checked
     * @return true if val is equal to MPI_MODE_NOPUT
     */
    virtual bool isModeNoput(int val) = 0;

    /**
     * Returns the constant for MPI_MODE_NOPUT.
     * @return value of MPI_MODE_NOPUT.
     */
    virtual int getModeNoput(void) = 0;

    /**
     * Checks if an argument is equal to MPI_MODE_NOPRECEDE
     * @param val value that is checked
     * @return true if val is equal to MPI_MODE_NOPRECEDE
     */
    virtual bool isModeNoprecede(int val) = 0;

    /**
     * Returns the constant for MPI_MODE_NOPRECEDE.
     * @return value of MPI_MODE_NOPRECEDE.
     */
    virtual int getModeNoprecede(void) = 0;

    /**
     * Checks if an argument is equal to MPI_MODE_NOSUCCEED
     * @param val value that is checked
     * @return true if val is equal to MPI_MODE_NOSUCCEED
     */
    virtual bool isModeNosucceed(int val) = 0;

    /**
     * Returns the constant for MPI_MODE_NOSUCCEED.
     * @return value of MPI_MODE_NOSUCCEED.
     */
    virtual int getModeNosucceed(void) = 0;

    /**
     * Returns the constant for MPI_LOCK_EXCLUSIVE.
     * @return value of MPI_LOCK_EXCLUSIVE.
     */
    virtual int getLockExclusive(void) = 0;

    /**
     * Checks if an argument is equal to MPI_LOCK_SHARED
     * @param val value that is checked
     * @return true if val is equal to MPI_LOCK_SHARED
     */
    virtual bool isLockShared(int val) = 0;

    /**
     * Returns the constant for MPI_LOCK_SHARED.
     * @return value of MPI_LOCK_SHARED.
     */
    virtual int getLockShared(void) = 0;

    /**
     * Checks if an argument is equal to MPI_MODE_NOCHECK
     * @param val value that is checked
     * @return true if val is equal to MPI_MODE_NOCHECK
     */
    virtual bool isModeNocheck(int val) = 0;

    /**
     * Checks if an argument is equal to MPI_LOCK_EXCLUSIVE
     * @param val value that is checked
     * @return true if val is equal to MPI_LOCK_EXCLUSIVE
     */
    virtual bool isLockExclusive(int val) = 0;

}; /*class I_BaseConstants*/

#endif /*I_BASECONSTANTS_H*/
