/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_WinEpoch.h
 *       @see I_WinEpoch.
 *
 */

#include "MustTypes.h"
#include "BaseIds.h"
#include <mpi.h>
#include <map>
#include <unordered_set>

#ifndef I_WINEPOCH_H
#define I_WINEPOCH_H

namespace must
{
/**
 * Enumeration of the types of window synchronizations.
 */
enum MUST_WIN_EPOCH_SYNC {
    MUST_WIN_EPOCH_NONE = 0,            /**< No active access epoch*/
    MUST_WIN_EPOCH_POTENTIAL_FENCE = 1, /**< Access and exposure epoch that will be created by
                                           MPI_Win_fence if followed by an RMA call.*/
    MUST_WIN_EPOCH_FENCE = 2,           /**< Access and exposure epoch created by MPI_Win_fence.*/
    MUST_WIN_EPOCH_LOCK = 4,            /**< Access Epoch created with MPI_Win_lock.*/
    MUST_WIN_EPOCH_LOCK_ALL = 8,        /**< Access Epoch created with MPI_Win_lock_all.*/
    MUST_WIN_EPOCH_POST = 16,           /**< Access and exposure Epoch created with MPI_Win_post.*/
    MUST_WIN_EPOCH_START = 32           /**< Access and access Epoch created with MPI_Win_start.*/
};

/**
 * Masks for access and exposure epochs.
 */
enum MUST_WIN_EPOCH_MASK {
    MUST_WIN_ACCESS_EPOCH_MASK = 0b00101111,  // POTENTIAL_FENCE, FENCE, LOCK, LOCK_ALL, START
    MUST_WIN_EXPOSURE_EPOCH_MASK = 0b00010011 // POTENTIAL_FENCE, FENCE, POST
};

class I_WinEpoch
{
  public:
    virtual unsigned int getSync(void) const = 0;
    virtual MustLocationId getEpochLId(int key) const = 0;
    virtual std::map<int, MustLocationId>& getEpochLIdMap(void) = 0;
    virtual std::unordered_set<int>& getWinStartRanks(void) = 0;
    virtual MustLocationId getLastRMALId(void) const = 0;
    virtual int getAssertion(void) const = 0;
    virtual int getEpochLIdKey(MUST_WIN_EPOCH_SYNC sync, int rank) const = 0;

}; /*class I_WinEpoch*/

} /*namespace must*/

#endif /*I_WINEPOCH_H*/
