/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef TSANMESSAGES_H
#define TSANMESSAGES_H

/**
 * @file TSanMessages.h
 *       @see MUST::TSanMessages.
 *
 *  @date 23.11.2017
 *  @author Felix Dommes
 */

#include "BaseApi.h"
#include "I_CreateMessage.h"
#include "I_GenerateLocationId.h"
#include "I_ParallelIdAnalysis.h"
#include "I_TSanMessages.h"
#include "ModuleBase.h"
#include "I_TargetRaceVerifier.h"
#include "LocationInfo.h"

#include "tsan_report_desc.hpp"
namespace __tsan
{
class ReportDesc;
}

namespace must
{

struct DelayedLocationInfo {
    LocationInfo locInfo;
    int stack_levels;
    std::vector<int> stack_indices;
    std::string stack_string;
    MustLocationId lId;
    MustParallelId pId;
};

struct DelayedMesssage {
    MustMessageIdNames msgId;
    MustParallelId pId;
    MustLocationId lId;
    MustMessageType msgType;
    std::string text;
    std::list<std::pair<MustParallelId, MustLocationId>> refLocations;
};

/**
 * Adapter to pipe TSan messages into MUST-messages
 */
class TSanMessages : public gti::ModuleBase<TSanMessages, I_TSanMessages>
{
  public:
    /**
     * Constructor.
     * @param instanceName name of this module instance.
     */
    explicit TSanMessages(const char* instanceName);

    /**
     * Destructor.
     */
    ~TSanMessages() override;

    /**
     * Creates messages from TSan reports.
     *
     * @param report pointer to the Thread Sanitizer report
     * @return see gti::GTI_ANALYSIS_RETURN.
     */
    template <__tsan::TSanVersion T>
    gti::GTI_ANALYSIS_RETURN tsanReport(const __tsan::ReportDescT<T>* report);
    template <__tsan::TSanVersion T>
    gti::GTI_ANALYSIS_RETURN tsanReportRMA(const __tsan::ReportDescT<T>* report);

    template <__tsan::TSanVersion T>
    void createLocation(
        MustParallelId pId,
        const __tsan::SymbolizedStack<T>* const stack,
        MustLocationId& lId);

    gti::GTI_ANALYSIS_RETURN fini() override;

  private:
    template <__tsan::TSanVersion T>
    std::string formatRMA(
        const __tsan::ReportDescT<T>* const report,
        std::list<std::pair<MustParallelId, MustLocationId>>& refList);
    I_CreateMessage* myLogger;
    I_GenerateLocationId* myGenLId;
    I_ParallelIdAnalysis* myPIdMod;
    I_TargetRaceVerifier* myTargetRaceVerifierMod;
    handleNewLocationP myNewLocFunc;
    bool myDelayedReport{false};
    std::vector<DelayedLocationInfo> delayedLocations;
    std::vector<DelayedMesssage> delayedMessages;
};
} // namespace must

#endif /*TSANMESSAGES_H*/
