/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file I_WinChecks.h
 *       @see I_WinChecks.
 *
 *  @date 14.11.2024
 *  @author Cornelius Pätzold
 */

#include "I_Module.h"
#include "GtiEnums.h"
#include "BaseIds.h"
#include "MustTypes.h"

#ifndef I_WINCHECKS_H
#define I_WINCHECKS_H

class I_WinChecks : public gti::I_Module
{
  public:
    virtual gti::GTI_ANALYSIS_RETURN warningIfWindowsOverlapCreate(
        MustParallelId pId,
        MustLocationId lId,
        void* base,
        int size,
        int kind) = 0;
    virtual gti::GTI_ANALYSIS_RETURN
    warningIfWindowsOverlap(MustParallelId pId, MustLocationId lId, MustWinType win) = 0;
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfWinAttachOverlap(MustParallelId pId, MustLocationId lId, MustWinType win) = 0;

    virtual gti::GTI_ANALYSIS_RETURN errorIfConflictingActiveEpoch(
        MustParallelId pId,
        MustLocationId lId,
        int syncKind,
        int target_rank,
        MustWinType win) = 0;
    virtual gti::GTI_ANALYSIS_RETURN errorIfRMACallOutsideEpoch(
        MustParallelId pId,
        MustLocationId lId,
        int target_rank,
        MustWinType win) = 0;
    virtual gti::GTI_ANALYSIS_RETURN errorIfFenceAssertNoPrecede(
        MustParallelId pId,
        MustLocationId lId,
        int assert,
        MustWinType win) = 0;
    virtual gti::GTI_ANALYSIS_RETURN errorIfFenceAssertNoSucceed(
        MustParallelId pId,
        MustLocationId lId,
        int target_rank,
        MustWinType win) = 0;
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfOpenEpochOnWinFree(MustParallelId pId, MustLocationId lId, MustWinType win) = 0;
    virtual gti::GTI_ANALYSIS_RETURN errorIfUnlockingWrongRank(
        MustParallelId pId,
        MustLocationId lId,
        int target_rank,
        MustWinType win) = 0;

    /**
     *	Checks if the window is known, if not,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the window to check.
     *	@param win window of the call.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfWinNotKnown(MustParallelId pId, MustLocationId lId, int aId, MustWinType win) = 0;

    /**
     *	Checks if the window is MPI_WIN_NULL, if so,
     *	manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the window to check.
     *	@param comm window of the call.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN
    errorIfWinNull(MustParallelId pId, MustLocationId lId, int aId, MustWinType win) = 0;

    /**
     *	Checks if an integer value (rank) is greater then or equal to the size of the window's
     *  communicator, manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the integer to check.
     *	@param value to check.
     *	@param win window of the call.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN errorIfRankGreaterEqualWinSize(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        int value,
        MustWinType win) = 0;

    /**
     *	Checks if a rank in a group is in the windows's communicator, manifests as error.
     *
     *	@param pId parallel Id of the call site.
     *	@param lId location Id of the call site.
     *	@param aId argument Id of the integer to check.
     *	@param group to check.
     *	@param win window of the call.
     *	@return see gti::GTI_ANALYSIS_RETURN.
     */
    virtual gti::GTI_ANALYSIS_RETURN errorIfRankInGroupNotInWin(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        MustGroupType group,
        MustWinType win) = 0;
};

#endif /*I_WINCHECKS_H*/