/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file WinChecks.h
 *       @see WinChecks.
 *
 *  @date 14.11.2024
 *  @author Cornelius Pätzold
 */

#include "ModuleBase.h"
#include "I_ParallelIdAnalysis.h"
#include "I_BaseConstants.h"
#include "I_CreateMessage.h"
#include "I_WinTrack.h"
#include "I_WinChecks.h"
#include "MustDefines.h"
#include "StridedBlock.h"
#include "I_GroupTrack.h"
#include "I_ArgumentAnalysis.h"
#include <map>

#ifndef WINCHECKS_H
#define WINCHECKS_H

using namespace gti;

namespace must
{

class WinChecks : public gti::ModuleBase<WinChecks, I_WinChecks>
{
  public:
    /**
     * Constructor.
     * @param instanceName name of this module instance.
     */
    WinChecks(const char* instanceName);

    /**
     * Destructor.
     */
    virtual ~WinChecks(void);

    GTI_ANALYSIS_RETURN warningIfWindowsOverlapCreate(
        MustParallelId pId,
        MustLocationId lId,
        void* base,
        int size,
        int kind);
    GTI_ANALYSIS_RETURN
    warningIfWindowsOverlap(MustParallelId pId, MustLocationId lId, MustWinType win) override;
    GTI_ANALYSIS_RETURN
    errorIfWinAttachOverlap(MustParallelId pId, MustLocationId lId, MustWinType win) override;

    GTI_ANALYSIS_RETURN errorIfConflictingActiveEpoch(
        MustParallelId pId,
        MustLocationId lId,
        int syncKind,
        int target_rank,
        MustWinType win) override;
    GTI_ANALYSIS_RETURN errorIfRMACallOutsideEpoch(
        MustParallelId pId,
        MustLocationId lId,
        int target_rank,
        MustWinType win) override;
    GTI_ANALYSIS_RETURN
    errorIfFenceAssertNoPrecede(MustParallelId pId, MustLocationId lId, int assert, MustWinType win)
        override;
    GTI_ANALYSIS_RETURN errorIfFenceAssertNoSucceed(
        MustParallelId pId,
        MustLocationId lId,
        int target_rank,
        MustWinType win) override;
    GTI_ANALYSIS_RETURN
    errorIfOpenEpochOnWinFree(MustParallelId pId, MustLocationId lId, MustWinType win) override;
    GTI_ANALYSIS_RETURN errorIfUnlockingWrongRank(
        MustParallelId pId,
        MustLocationId lId,
        int target_rank,
        MustWinType win) override;

    GTI_ANALYSIS_RETURN
    errorIfWinNotKnown(MustParallelId pId, MustLocationId lId, int aId, MustWinType win) override;

    GTI_ANALYSIS_RETURN
    errorIfWinNull(MustParallelId pId, MustLocationId lId, int aId, MustWinType win) override;

    GTI_ANALYSIS_RETURN
    errorIfRankGreaterEqualWinSize(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        int value,
        MustWinType win) override;

    GTI_ANALYSIS_RETURN
    errorIfRankInGroupNotInWin(
        MustParallelId pId,
        MustLocationId lId,
        int aId,
        MustGroupType group,
        MustWinType win) override;

  protected:
    I_ParallelIdAnalysis* myPIdMod;
    I_BaseConstants* myConsts;
    I_CreateMessage* myLogger;
    I_WinTrack* myWinMod;
    I_GroupTrack* myGroupMod;
    I_ArgumentAnalysis* myArgMod;
};

} // namespace must

#endif /*WINCHECKS_H*/