/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file PrintFile.cpp
 *       @see MUST::PrintFile.
 *
 *  @date 12.05.2011
 *  @author Tobias Hilbrich
 */

#include "GtiMacros.h"
#include "PrintFile.h"

#include <assert.h>
#include <sstream>

using namespace must;

mGET_INSTANCE_FUNCTION(PrintFile);
mFREE_INSTANCE_FUNCTION(PrintFile);
mPNMPI_REGISTRATIONPOINT_FUNCTION(PrintFile);

//=============================
// Constructor
//=============================
PrintFile::PrintFile(const char* instanceName)
    : gti::ModuleBase<PrintFile, I_PrintFile>(instanceName), myLogger(NULL), myFileTracker(NULL),
      myLocations(NULL)
{
    //create sub modules
    std::vector<I_Module*> subModInstances;
    subModInstances = createSubModuleInstances();

    //handle sub modules
    if (subModInstances.size() < 3) {
        std::cerr << "error: " << __LINE__ << "@" << __FILE__
                  << " has not enough sub modules, aborting!" << std::endl;
        assert(0);
    }

    myLocations = (I_LocationAnalysis*)subModInstances[0];
    myLogger = (I_CreateMessage*)subModInstances[1];
    myFileTracker = (I_FileTrack*)subModInstances[2];
}

//=============================
// Destructor
//=============================
PrintFile::~PrintFile()
{
    if (myLocations)
        destroySubModuleInstance((I_Module*)myLocations);
    myLocations = NULL;

    if (myLogger)
        destroySubModuleInstance((I_Module*)myLogger);
    myLogger = NULL;

    if (myFileTracker)
        destroySubModuleInstance((I_Module*)myFileTracker);
    myFileTracker = NULL;
}

//=============================
// print
//=============================
GTI_ANALYSIS_RETURN PrintFile::print(MustParallelId pId, MustLocationId lId, MustFileType file)
{
    static int index = 0;
    std::stringstream stream;
    stream << "Information on file: ";
    std::list<std::pair<MustParallelId, MustLocationId>> refs;
    I_File* info = myFileTracker->getFile(pId, file);

    if (info)
        info->printInfo(stream, &refs);
    else
        stream << "Unknown File.";

    myLogger->createMessage(index++, pId, lId, MustInformationMessage, stream.str(), refs);

    return GTI_ANALYSIS_SUCCESS;
}

/*EOF*/
