/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file PrintInfo.cpp
 *       @see MUST::PrintInfo.
 *
 *  @date 12.05.2011
 *  @author Tobias Hilbrich
 */

#include "GtiMacros.h"
#include "PrintInfo.h"

#include <assert.h>
#include <sstream>

using namespace must;

mGET_INSTANCE_FUNCTION(PrintInfo);
mFREE_INSTANCE_FUNCTION(PrintInfo);
mPNMPI_REGISTRATIONPOINT_FUNCTION(PrintInfo);

//=============================
// Constructor
//=============================
PrintInfo::PrintInfo(const char* instanceName)
    : gti::ModuleBase<PrintInfo, I_PrintInfo>(instanceName), myLogger(NULL), myInfoTracker(NULL),
      myLocations(NULL)
{
    //create sub modules
    std::vector<I_Module*> subModInstances;
    subModInstances = createSubModuleInstances();

    //handle sub modules
    if (subModInstances.size() < 3) {
        std::cerr << "Error: " << __LINE__ << "@" << __FILE__
                  << " has not enough sub modules, aborting!" << std::endl;
        assert(0);
    }

    myLocations = (I_LocationAnalysis*)subModInstances[0];
    myLogger = (I_CreateMessage*)subModInstances[1];
    myInfoTracker = (I_InfoTrack*)subModInstances[2];
}

//=============================
// Destructor
//=============================
PrintInfo::~PrintInfo()
{
    if (myLocations)
        destroySubModuleInstance((I_Module*)myLocations);
    myLocations = NULL;

    if (myLogger)
        destroySubModuleInstance((I_Module*)myLogger);
    myLogger = NULL;

    if (myInfoTracker)
        destroySubModuleInstance((I_Module*)myInfoTracker);
    myInfoTracker = NULL;
}

//=============================
// print
//=============================
GTI_ANALYSIS_RETURN
PrintInfo::print(MustParallelId pId, MustLocationId lId, MustInfoType infoHandle)
{
    static int index = 0;
    std::stringstream stream;
    stream << "Information on key: ";
    std::list<std::pair<MustParallelId, MustLocationId>> refs;
    I_Info* info = myInfoTracker->getInfo(pId, infoHandle);

    if (info) {
        info->printInfo(stream, &refs);
    } else {
        stream << "Unknown Info.";
    }

    myLogger->createMessage(index++, pId, lId, MustInformationMessage, stream.str(), refs);

    return GTI_ANALYSIS_SUCCESS;
}

/*EOF*/
