/* Part of the MUST Project, under BSD-3-Clause License
 * See https://hpc.rwth-aachen.de/must/LICENSE for license information.
 * SPDX-License-Identifier: BSD-3-Clause
 */

// RUN: %compile-cc
// RUN: %must-run %mpiexec-numproc-flag 2 --must:layout \
// RUN: %builddir/tests/InfoTrack/layout.xml --must:analyses \
// RUN: %builddir/tests/InfoTrack/analysis_spec.xml \
// RUN: %t.exe 2>&1 \
// RUN: | %filecheck %s

/**
 * @file testInfoTrack2.c
 * A must test case for info tracking.
 *
 * @author Tobias Hilbrich
 *
 */

#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>
#include "mustFeaturetested.h"
// #include "mustTest.h"

// #define TEST_INFO(R) MPI_Initialized(&R)
#define TEST_INFO(I)                                                                               \
    do {                                                                                           \
        if (rank == 0) {                                                                           \
            fInfo = MUST_Comm_m2i(I);                                                              \
            MPI_Initialized((int*)&fInfo);                                                         \
        }                                                                                          \
    } while (0)

/**
 * Performs the following actions:
 * (Using any number of processes)
 *
 * 1) Initialize a comm
 * 2) Creates a new info
 * 3) Set and get info handle associated with comm
 * 4) Clean up
 */
int main(int argc, char** argv)
{
    int rank, size;

    MustInfoType fInfo;
    MPI_Info info = MPI_INFO_NULL, info2 = MPI_INFO_NULL;
    MPI_Comm comm = MPI_COMM_NULL;
    MPI_Group world_group = MPI_GROUP_NULL;

    MPI_Init(&argc, &argv);
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);
    MPI_Comm_size(MPI_COMM_WORLD, &size);

    printf("Ready: %d of %d tasks.\n", rank, size);

    //Enough tasks ?
    if (size < 1) {
        printf("Not enough tasks, need 1 at least.\n");
        MPI_Finalize();
        exit(1);
    }

    //==1) Initialize comm
    MPI_Comm_group(MPI_COMM_WORLD, &world_group);
    MPI_Comm_create(MPI_COMM_WORLD, world_group, &comm);

    //==2) Info create
    MPI_Info_create(&info); // implicit (ref=1)
    // CHECK: [MUST-REPORT] Reference 1: call MPI_Info_create@rank 0
    TEST_INFO(info);

    MPI_Info_set(info, "key1", "value1");
    MPI_Info_set(info, "key12", "value12");

    // CHECK: [MUST-REPORT] Reference 1: call MPI_Info_create@rank 0
    TEST_INFO(info);
    //==3_ Set and get info handle associated with comm
    MPI_Comm_set_info(comm, info);   // persistent info (ref=3)
    MPI_Comm_get_info(comm, &info2); // ref=4
    // CHECK: [MUST-REPORT] Reference 1: call MPI_Comm_get_info@rank 0
    TEST_INFO(info2);

    //==4) Clean up
    MPI_Comm_free(&comm);         // delete persistent (ref--)
    MPI_Group_free(&world_group); // delete persistent (ref--)

    MPI_Info_free(&info);
    MPI_Info_free(&info2);
    // CHECK: [MUST-REPORT] Information: from: call MPI_Initialized@0:
    // CHECK-SAME: Information on key: MPI_INFO_NULL
    TEST_INFO(info);
    // CHECK: [MUST-REPORT] Information: from: call MPI_Initialized@0:
    // CHECK-SAME: Information on key: MPI_INFO_NULL
    TEST_INFO(info2);

    printf("Signing off: %d of %d tasks.\n", rank, size);
    MPI_Finalize();
    return 0;
}
